
# ************************************************************************
# *
# * SYNAPTIC LABORATORIES CONFIDENTIAL
# * ----------------------------------
# *
# *  (c) 2017 Synaptic Laboratories Limited
# *  All Rights Reserved.
# *
# * NOTICE:  All information contained herein is, and remains
# * the property of Synaptic Laboratories Limited and its suppliers,
# * if any.  The intellectual and technical concepts contained
# * herein are proprietary to Synaptic Laboratories Limited
# * and its suppliers and may be covered by U.S. and Foreign Patents,
# * patents in process, and are protected by trade secret or copyright law.
# * Dissemination of this information or reproduction of this material
# * is strictly forbidden unless prior written permission is obtained
# * from Synaptic Laboratories Limited
# *
# * Modification of this file is strictly forbidden unless prior written
# * permission is obtained from Synaptic Laboratories Limited
#
#######################################################################################################################
#functions used by tcl script
#######################################################################################################################

#-------------------------------------------------------------------------------
#log2ceil
#-------------------------------------------------------------------------------
proc log2ceil {num} {

    set val 0
    set i 1
    while {$i < $num} {
        set val [expr $val + 1]
        set i [expr 1 << $val]
    }

    return $val;
}

#-------------------------------------------------------------------------------
#proc set ranges for Memory memories
#-------------------------------------------------------------------------------
proc proc_set_memory_ranges {} {

   for {set dev 0} {$dev < 2} {incr dev} {

      #Get Protocol For Device
      set v_dev_protocol [get_parameter_value  GUI_DEV${dev}_PROTOCOL_MAN]

      switch $v_dev_protocol {
          "none"        {set_parameter_property GUI_DEV${dev}_MODEL ALLOWED_RANGES  "none:none"}
          "Generic_x8"  {set_parameter_property GUI_DEV${dev}_MODEL ALLOWED_RANGES  {\
                          "generic_x8_256m:Generic x8 Memory Device : 256 mbit address space"\
                          "generic_x8_512m:Generic x8 Memory Device : 512 mbit address space"\
                          "generic_x8_1g:Generic x8 Memory Device   : 1 Gbit address space"\
                          "generic_x8_2g:Generic x8 Memory Device   : 2 Gbit address space" }}
          "HyperFlash"  {set_parameter_property GUI_DEV${dev}_MODEL ALLOWED_RANGES { \
                          "s26ks128s:S26KS128S (Cypress) 1.8V HyperFlash 166 MHz 128 Mbit"\
                          "s26ks256s:S26KS256S (Cypress) 1.8V HyperFlash 166 MHz 256 Mbit"\
                          "s26ks512s:S26KS512S (Cypress) 1.8V HyperFlash 166 MHz 512 Mbit"\
                          "is26ks128s:IS26KS128S (ISSI)  1.8V HyperFlash 166 MHz 128 Mbit"\
                          "is26ks256s:IS26KS256S (ISSI)  1.8V HyperFlash 166 MHz 256 Mbit"\
                          "is26ks512s:IS26KS512S (ISSI)  1.8V HyperFlash 166 MHz 512 Mbit" \
                          "s26kl128s:S26KL128S 3.0V (Cypress) HyperFlash 133 MHz 128 Mbit"\
                          "s26kl256s:S26KL256S (Cypress) 3.0V HyperFlash 133 MHz 256 Mbit"\
                          "s26kl512s:S26KL512S (Cypress) 3.0V HyperFlash 133 MHz 512 Mbit"\
                          "is26kl128s:IS26KL128S (ISSI)  3.0V HyperFlash 133 MHz 128 Mbit"\
                          "is26kl256s:IS26KL256S (ISSI)  3.0V HyperFlash 133 MHz 256 Mbit"\
                          "is26kl512s:IS26KL512S (ISSI)  3.0V HyperFlash 133 MHz 512 Mbit"} }
          "SemperFlash"  {set_parameter_property GUI_DEV${dev}_MODEL ALLOWED_RANGES { \
                          "s26hs512t:S26HS512T (Cypress) 1.8V HBUS SemperFlash 200 MHz 512 Mbit"\
                          "s26hs01gt:S26HS01GT (Cypress) 1.8V HBUS SemperFlash 200 MHz 1   Gbit"\
                          "s26hs02gt:S26HS02GT (Cypress) 1.8V HBUS SemperFlash 200 MHz 2   Gbit"\
                          "s26hl512t:S26HL512T (Cypress) 3.0V HBUS SemperFlash 133 MHz 512 Mbit"\
                          "s26hl01gt:S26HL01GT (Cypress) 3.0V HBUS SemperFlash 133 MHz 1   Gbit"\
                          "s26hl02gt:S26HL02GT (Cypress) 3.0V HBUS SemperFlash 133 MHz 2   Gbit"\
                          "s28hs512t:S28HS512T (Cypress) 1.8V xSPI SemperFlash 200 MHz 512 Mbit"  \
                          "s28hs01gt:S28HS01GT (Cypress) 1.8V xSPI SemperFlash 200 MHz 1   Gbit"  \
                          "s28hs02gt:S28HS02GT (Cypress) 1.8V xSPI SemperFlash 200 MHz 2   Gbit"  \
                          "s28hl512t:S28HL512T (Cypress) 3.0V xSPI SemperFlash 133 MHz 512 Mbit"  \
                          "s28hl01gt:S28HL01GT (Cypress) 3.0V xSPI SemperFlash 133 MHz 1   Gbit"  \
                          "s28hl02gt:S28HL02GT (Cypress) 3.0V xSPI SemperFlash 133 MHz 2   Gbit" } }
          "HyperRAM"    {set_parameter_property GUI_DEV${dev}_MODEL ALLOWED_RANGES { \
                          "s27ks0641:S27KS0641 (Cypress) 1.8V HyperRAM 166 MHz 64 Mbit"\
                          "s70ks1281:S70KS1281 (Cypress) 1.8V HyperRAM 166 MHz 128 Mbit"\
                          "is66wvh08m8_all:IS66WVH8M8 ALL  (ISSI) 1.8V HyperRAM 166 MHz 64 Mbit"\
                          "is66wvh16m8_all:IS66WVH16M8 ALL (ISSI) 1.8V HyperRAM 166 MHz 128 Mbit"\
                          "is66wvh32m8_all:IS66WVH32M8 ALL (ISSI) 1.8V HyperRAM 166 MHz 256 Mbit"
                          "w955d8mky:W955D8MKY (Winbond) 1.8V HyperRAM 166 MHz 32 Mbit"\
                          "w956d8mky:W956D8MKY (Winbond) 1.8V HyperRAM 200 MHz 64 Mbit"\
                          "w957d8mky:W957D8MKY (Winbond) 1.8V HyperRAM 200 MHz 128 Mbit"\
                          "w958d6nky:W958D6NKY (Winbond) 1.8V HyperRAM 200 MHz 256 Mbit : 16-bit"\
                          "s27ks0642:S27KS0642 (Cypress) 1.8V HyperRAM 200 MHz 64 Mbit"\
                          "s27ks1282:S27KS01282 (Cypress) 1.8V  HyperRAM 200 MHz 128 Mbit"\
                          "s27kl0641:S27KL0641 (Cypress) 3.0V HyperRAM 100 MHz 64 Mbit"\
                          "s70kl1281:S70KL1281 (Cypress) 3.0V HyperRAM 100 MHz 128 Mbit"\
                          "is66wvh08m8_bll:IS66WVH8M8 BALL (ISSI) 3.0V HyperRAM 100 MHz 64 Mbit"\
                          "is66wvh16m8_bll:IS66WVH16M8 BALL (ISSI) 3.0V HyperRAM 100 MHz 128 Mbit"\
                          "is66wvh32m8_bll:IS66WVH32M8 BALL (ISSI) 3.0V HyperRAM 100 MHz 256 Mbit"
                          "w956a8mby:W956A8MBY  (Winbond) 3.0V HyperRAM 166 MHz 64 Mbit"\
                          "w957a8mby:W957A8MBY  (Winbond) 3.0V HyperRAM 166 MHz 128 Mbit"\
                          "s27kl0642:S27KS0642  (Cypress) 3.0V HyperRAM 166 MHz 64 Mbit"\
                          "s27kl1282:S27KS01282 (Cypress) 3.0V HyperRAM 166 MHz 128 Mbit" } }
          "OctaRAM"     {set_parameter_property GUI_DEV${dev}_MODEL ALLOWED_RANGES  {\
                          "jsc64ssp8ag:JSC64SSP8AG 1.8V (JSC) OctaRAM 200 MHz 64 Mbit"\
                          "jsc28ssp8ag:JSC28SSP8AG 1.8V (JSC) OctaRAM 200 MHz 128 Mbit"\
                          "ap6408l_oc:APS6408L OC 1.8V (AP) OctaRAM 200 Mhz 64  Mbit"\
                          "ap12808l_oc:APS12808L OC 1.8V (AP) OctaRAM 200 Mhz 128  Mbit"\
                          "is66wvo16m8_all:is66wvo16m8 ALL OC 1.8V (ISSI) OctaRAM 200 Mhz 128 Mbit"\
                          "is66wvo32m8_all:is66wvo32m8 ALL OC 1.8V (ISSI) OctaRAM 200 Mhz 256 Mbit"\
                          "jsc64ssu8ag:JSC64SSU8AG 3.0V (JSC) OctaRAM 133 MHz 64 Mbit"\
                          "jsc28ssu8ag:JSC28SSU8AG 3.0V (JSC) OctaRAM 133 MHz 128 Mbit"\
                          "ap6408l_3oc:APS6408L 3OC 3.0V (AP) OctaRAM 133 Mhz 64  Mbit"\
                          "ap12808l_3oc_3v:APS12808L 3OC 3.0V  (AP) OctaRAM 133 Mhz 128  Mbit" \
                          "is66wvo16m8_bll:is66wvo16m8 BLL 3OC 3.0V (ISSI) OctaRAM 133 Mhz 128 Mbit"\
                          "is66wvo32m8_bll:is66wvo32m8 ALL 3OC 3.0V  (ISSI) OctaRAM 133 Mhz 256 Mbit"} }
          "OctaFlash"     {set_parameter_property GUI_DEV${dev}_MODEL ALLOWED_RANGES  {\
                          "mx25lu512:MX25LU512 (Matronix) 1.8V OctaFlash 200 MHz 512 Mbit" \
                          "mx25lm512:MX25LM512 (Matronix) 3.0V OctaFlash 133 MHz 512 Mbit"} }
          "XccelaRAM"   {set_parameter_property GUI_DEV${dev}_MODEL ALLOWED_RANGES { \
                          "ap1604m:APS1604M OB   (AP) 1.8V XcellaRAM 166Mhz 16 Mbit 4-bit" \
                          "ly68s6408:LY68S6408 (LyonTek) 1.8V XccelaRAM 166 Mhz 64 Mbit" \
                          "ly68x6408:LY68X6408 (LyonTek) 1.8V XccelaRAM 200 Mhz 64 Mbit" \
                          "ap3208l:APS3208L OB   (AP) 1.8V XccelaRAM 200 Mhz 32  Mbit "\
                          "ap6408l:APS6408L OB   (AP) 1.8V XccelaRAM 200 Mhz 64  Mbit "\
                          "ap12808l:APS12808L OB (AP) 1.8V XccelaRAM 200 Mhz 128 Mbit "\
                          "ap25608l:APS25608L OB (AP) 1.8V XccelaRAM 200 Mhz 256 Mbit "\
                          "ap51208l:APS51208L OB (AP) 1.8V XccelaRAM 200 Mhz 512 Mbit "\
                          "ly68l6408:LY68L6408 (LyonTek) 3.0V XccelaRAM 64 Mbit 133Mhz " \
                          "ap3208l_3ob:APS3208L 3OB (AP) 3.0V XccelaRAM 133 Mhz 32  Mbit "\
                          "ap6408l_3ob:APS6408L 3OB (AP) 3.0V XccelaRAM 133 Mhz 64  Mbit "\
                          "ap12808l_3ob:APS12808L 3OB (AP) 3.0V XccelaRAM 133 Mhz 128 Mbit "\
                          "ap25608l_3ob:APS25608L 3OB (AP) 3.0V XccelaRAM 133 Mhz 256 Mbit "\
                          "ap51208l_3ob:APS51208L 3OB (AP) 3.0V XccelaRAM 133 Mhz 512 Mbit "\
                          "ap256016l:APS25608L OB (AP) 1.8V XccelaRAM 200 Mhz 256 Mbit : 16-bit "\
                          "ap512016l:APS51208L OB (AP) 1.8V XccelaRAM 200 Mhz 512 Mbit : 16-bit"\
                          "ap256016l_3ob:APS25608L 3OB (AP) 3.0V XccelaRAM 133 Mhz 256 Mbit : 16-bit "\
                          "ap512016l_3ob:APS51208L 3OB (AP) 3.0V XccelaRAM 133 Mhz 512 Mbit : 16-bit" }}
          "xSPI_PSRAM"  {set_parameter_property GUI_DEV${dev}_MODEL ALLOWED_RANGES  { \
                         "s27kl0643:S27KL0643 (Cypress) 3.0V xSPI HyperRAM 133 MHz 64  Mbit"\
                         "s27kl1283:S27KL1283 (Cypress) 3.0V xSPI HyperRAM 133 MHz 128 Mbit"\
                         "s27ks0643:S27KS0643 (Cypress) 1.8V xSPI HyperRAM 200 MHz 64 Mbit" \
                         "s27ks1283:S27KS1283 (Cypress) 1.8V xSPI HyperRAM 200 MHz 128 Mbit" }}
        "xSPI STT-MRAM" {set_parameter_property GUI_DEV${dev}_MODEL ALLOWED_RANGES  { \
                         "em064lx:EM064LX   (Everspin) 1.8V  xSPI STT-MRAM 200 MHz 64 Mbit" }}
          "xSPI_Flash"   {set_parameter_property GUI_DEV${dev}_MODEL ALLOWED_RANGES  { \
                         "mt35xu256:MT35XU256   (Micron) 1.8V  xSPI XccelaFlash 200 MHz 256 Mbit" \
                         "mt35xu512:MT35XU512   (Micron) 1.8V  xSPI XccelaFlash 200 MHz 512 Mbit" \
                         "mt35xu01g:MT35XU01G   (Micron) 1.8V  xSPI XccelaFlash 200 MHz 1024 Mbit" \
                         "mt35xu02g:MT35XU02G   (Micron) 1.8V  xSPI XccelaFlash 200 MHz 2048 Mbit" \
                         "is25wo128:IS25LO(128) (ISSI)   1.8V  xSPI XccelaFlash 200 MHz 128 Mbit" \
                         "is25wo256:IS25LO(256) (ISSI)   1.8V  xSPI XccelaFlash 200 MHz 256 Mbit" \
                         "is25wo512:IS25LO(512) (ISSI)   1.8V  xSPI XccelaFlash 200 MHz 512 Mbit" \
                         "gd55lx512:GD55LX512   (GigaDevice) 1.8V  xSPI XccelaFlash 200 MHz 512 Mbit" \
                         "gd55lx01g:GD55LX01G   (GigaDevice) 1.8V  xSPI XccelaFlash 200 MHz 1024 Mbit" \
                         "gd55lx02g:GD55LX02G   (GigaDevice) 1.8V  xSPI XccelaFlash 200 MHz 2047 Mbit" \
                         "mt35xl256:MT35XL256   (Micron) 3.0V  xSPI XccelaFlash 133 MHz 256 Mbit" \
                         "mt35xl512:MT35XL512   (Micron) 3.0V  xSPI XccelaFlash 133 MHz 512 Mbit" \
                         "mt35xl01g:MT35XL01G   (Micron) 3.0V  xSPI XccelaFlash 133 MHz 1024 Mbit" \
                         "mt35xl02g:MT35XL02G   (Micron) 3.0V  xSPI XccelaFlash 133 MHz 2048 Mbit" \
                         "is25lo128:IS25LO(128) (ISSI)   3.0V  xSPI XccelaFlash 133 MHz 128 Mbit" \
                         "is25lo256:IS25LO(256) (ISSI)   3.0V  xSPI XccelaFlash 133 MHz 256 Mbit" \
                         "is25lo512:IS25LO(512) (ISSI)   3.0V  xSPI XccelaFlash 133 MHz 512 Mbit"  }}
          default       {set_parameter_property GUI_DEV${dev}_MODEL ALLOWED_RANGES  "none:none"}
      }
   }
}


#-------------------------------------------------------------------------------
#proc get timing cycles (round up to next integer)
#-------------------------------------------------------------------------------
proc proc_calc_timing {v_tim_param clock_speed_mhz} {
     set v_tim_param_cycles [expr ${v_tim_param} * ${clock_speed_mhz}]

       if {[expr fmod($v_tim_param_cycles, 1000)] != 0 } {
          set v_tim_param_cycles [expr ($v_tim_param_cycles/1000) + 1]
       } else {
          set v_tim_param_cycles [expr ($v_tim_param_cycles)/1000]
       }
   return ${v_tim_param_cycles}
}


#-------------------------------------------------------------------------------
# Generate timing parameter
#-------------------------------------------------------------------------------
proc proc_get_dev_timing {v_dev1_timing v_dev0_timing } {

  #device not mapped
  if {$v_dev0_timing == 0}  {

    set dev_timing $v_dev1_timing

  } else {
    set  dev1_timing_tacc  [expr $v_dev1_timing & 0x00ff0000]
    set  dev1_timing_trwr  [expr $v_dev1_timing & 0x0000ff00]
    set  dev1_timing_tcsh  [expr $v_dev1_timing & 0x000000f0]
    set  dev1_timing_tcss  [expr $v_dev1_timing & 0x0000000f]

    set  dev0_timing_tacc  [expr $v_dev0_timing & 0x00ff0000]
    set  dev0_timing_trwr  [expr $v_dev0_timing & 0x0000ff00]
    set  dev0_timing_tcsh  [expr $v_dev0_timing & 0x000000f0]
    set  dev0_timing_tcss  [expr $v_dev0_timing & 0x0000000f]

    #Set Tacc for Device 0, Device 1
    set  dev_timing        [expr ($dev1_timing_tacc << 8) | $dev0_timing_tacc]

    #Select Select Greater Value of  Trwr
    if {$dev1_timing_trwr > $dev0_timing_trwr} {
      set  dev_timing      [expr $dev_timing | $dev1_timing_trwr]
    } else {
      set  dev_timing      [expr $dev_timing | $dev0_timing_trwr]
    }

    #Select Select Greater Value of  Tcsh
    if {$dev1_timing_tcsh > $dev0_timing_tcsh} {
      set  dev_timing      [expr $dev_timing | $dev1_timing_tcsh]
    } else {
      set  dev_timing      [expr $dev_timing | $dev0_timing_tcsh]
    }

    #Select Greater Value of Tcss
    if {$dev1_timing_tcss > $dev0_timing_tcss} {
      set  dev_timing      [expr $dev_timing | $dev1_timing_tcss]
    } else {
      set  dev_timing      [expr $dev_timing | $dev0_timing_tcss]
    }

  }

 return  $dev_timing
}

#-------------------------------------------------------------------------------
#set display group
#-------------------------------------------------------------------------------
proc proc_add_master_configuration  {} {

add_display_item ""                                         "Master Configuration"                              GROUP tab

add_display_item "Master Configuration"                     "Memory Channel Configuration"                    GROUP ""
add_display_item "Master Configuration"                     "Memory Channel 0 Configuration"                  GROUP ""
add_display_item "Master Configuration"                     "Memory Channel 1 Configuration"                  GROUP ""
add_display_item "Master Configuration"                     "Memory IO Configuration"                         GROUP ""
add_display_item "Master Configuration"                     "Memory Powerup Configuration"                    GROUP ""
add_display_item "Master Configuration"                     "Memory RWDS Configuration"                       GROUP ""
add_display_item "Master Configuration"                     "Memory RWDS-DQ SKEW Configuration"               GROUP ""
add_display_item "Master Configuration"                     "Avalon Target Interface Configuration"           GROUP ""
add_display_item "Master Configuration"                     "Derived Parameters"                              GROUP COLLAPSED

add_display_item "Memory Channel Configuration"             GUI_HBUS_CHANNELS                    PARAMETER "" ;# FPGA board
add_display_item "Memory Channel Configuration"             GUI_NUM_CHIPSELECTS                   PARAMETER "" ;# Number of Chip Select
add_display_item "Memory Powerup Configuration"             GUI_POWERUP_TIMER                     PARAMETER "" ;# Number of Chip Select

add_display_item "Memory IO Configuration"                  GUI_DQIN_MODE                         PARAMETER "" ;# FPGA board
add_display_item "Memory IO Configuration"                  GUI_PYHLITE_INTERFACE                 PARAMETER "" ;# FPGA board
add_display_item "Memory IO Configuration"                  GUI_PYHLITE_ADD_LATENCY               PARAMETER "" ;# FPGA board

add_display_item "Memory RWDS-DQ SKEW Configuration"        GUI_DQ_DQS_SKEW_DEFAULT              PARAMETER "" ;# Include register Avalon port

add_display_item "Avalon Target Interface Configuration"    GUI_INCLUDE_REG_AVALON                PARAMETER "" ;# Include register Avalon port
add_display_item "Avalon Target Interface Configuration"    GUI_REG_APB_CONTROL_PORT             PARAMETER "" ;# Include register Avalon port

add_display_item "Avalon Target Interface Configuration"    GUI_INCLUDE_HBUS_ECC                  PARAMETER "" ;# Include register Avalon port

proc_add_parameter      GUI_HBUS_CHANNELS           INTEGER 1 "" false "Memory Channel Configuration"
set_parameter_property  GUI_HBUS_CHANNELS           DISPLAY_NAME  "Number of Parallel Memory channels"
set_parameter_property  GUI_HBUS_CHANNELS           ALLOWED_RANGES {1}
set_parameter_property  GUI_HBUS_CHANNELS           DESCRIPTION  "Number of Parallel Memory channels"
set_parameter_property  GUI_HBUS_CHANNELS           ENABLED        true


proc_add_parameter      GUI_NUM_CHIPSELECTS         INTEGER 2 "" true "Memory Channel Configuration"
set_parameter_property  GUI_NUM_CHIPSELECTS         DISPLAY_NAME  "Number of chip selects on this memory channel"
set_parameter_property  GUI_NUM_CHIPSELECTS         ALLOWED_RANGES {2}
set_parameter_property  GUI_NUM_CHIPSELECTS         ENABLED        false
set_parameter_property  GUI_NUM_CHIPSELECTS         VISIBLE        false
set_parameter_property  GUI_NUM_CHIPSELECTS         DESCRIPTION \
    "The number of chip selects signals driven by this MBMC is always 2.\
     Chip selects wih no devices will be constantly driven by the MBMC as inactive to ensure correct operation."


proc_add_parameter      GUI_DQIN_MODE INTEGER 1 "" false  "Memory IO Configuration"
set_parameter_property  GUI_DQIN_MODE DISPLAY_NAME   "Select DQIN Pin configuration"
set_parameter_property  GUI_DQIN_MODE ALLOWED_RANGES {"0 : Direct connection" \
                                                      "1 : Integrated DDR Registers in IO Element " }
set_parameter_property  GUI_DQIN_MODE      ENABLED    true

proc_add_parameter      GUI_PYHLITE_INTERFACE BOOLEAN false "" false  "Memory IO Configuration"
set_parameter_property  GUI_PYHLITE_INTERFACE DISPLAY_NAME    "Select I/O Interface"
set_parameter_property  GUI_PYHLITE_INTERFACE ALLOWED_RANGES {"false: Standad I/O Interface" \
                                                              "true:  PhyLite Interface" }

proc_add_parameter      GUI_PYHLITE_ADD_LATENCY INTEGER 1 "" false  "Memory IO Configuration"
set_parameter_property  GUI_PYHLITE_ADD_LATENCY DISPLAY_NAME   "Phylite Additional Read Latency Cycle"
set_parameter_property  GUI_PYHLITE_ADD_LATENCY ALLOWED_RANGES {0 1}

proc_add_parameter      GUI_DQ_DQS_SKEW_DEFAULT  BOOLEAN true "" false  "Memory RWDS-DQ SKEW Configuration"
set_parameter_property  GUI_DQ_DQS_SKEW_DEFAULT  DISPLAY_NAME "Enable Default RWDS-DQ SKEW settings" ;
set_parameter_property  GUI_DQ_DQS_SKEW_DEFAULT  DESCRIPTION  "Enable Default RWDS-DQ SKEW settings"

proc_add_parameter      GUI_POWERUP_TIMER INTEGER 150 "microseconds" false  "Memory Powerup Configuration"
set_parameter_property  GUI_POWERUP_TIMER DISPLAY_NAME   "PowerUp Delay Timer"

###==========================================================
### Start Backwards compatability v2.12
###
proc_add_parameter      GUI_INCLUDE_REG_AVALON      INTEGER false "" false "Avalon Target Interface Configuration"
set_parameter_property  GUI_INCLUDE_REG_AVALON      DISPLAY_NAME  "Avalon control port"
set_parameter_property  GUI_INCLUDE_REG_AVALON      ALLOWED_RANGES {"0:Disabled" "1:Enabled"}
set_parameter_property  GUI_INCLUDE_REG_AVALON      DESCRIPTION   "\
      Add an Avalon Target Interface to provide access to the configuration registers of this HMBC instance."
set_parameter_property  GUI_INCLUDE_REG_AVALON      VISIBLE        true

proc_add_parameter      GUI_REG_APB_CONTROL_PORT    INTEGER false "" false "Avalon Target Interface Configuration"
set_parameter_property  GUI_REG_APB_CONTROL_PORT     DISPLAY_NAME  "Control Port Interface Type"
set_parameter_property  GUI_REG_APB_CONTROL_PORT     ALLOWED_RANGES {"0:AVMM Interface"}
##set_parameter_property  GUI_REG_APB_CONTROL_PORT     ALLOWED_RANGES {"0:AVMM Interface" "1:APB Interface"}
set_parameter_property  GUI_REG_APB_CONTROL_PORT     VISIBLE        true


proc_add_parameter      GUI_INCLUDE_HBUS_ECC      INTEGER false "" false "Avalon Target Interface Configuration"
set_parameter_property  GUI_INCLUDE_HBUS_ECC      DISPLAY_NAME  "Include PSRAM ECC input signal"
set_parameter_property  GUI_INCLUDE_HBUS_ECC      ALLOWED_RANGES {"0:Disabled" "1:Enabled"}
set_parameter_property  GUI_INCLUDE_HBUS_ECC      DESCRIPTION   "Include PSRAM ECC input signal"
set_parameter_property  GUI_INCLUDE_HBUS_ECC      VISIBLE        true


#---------------------------------------------------
# Add Channels
#---------------------------------------------------

   for {set dev 0} {$dev < 2} {incr dev} {

       # Instantiate Manual Configuration
       #   The Value of the Manual configuration will be loaded into the AUTO configuration as required.
       #
       add_display_item "Memory Channel $dev Configuration"  GUI_DEV${dev}_PROTOCOL_MAN  PARAMETER ""

       proc_add_parameter     GUI_DEV${dev}_PROTOCOL_MAN String "none" "" false "Memory Channel $dev Configuration"
       set_parameter_property GUI_DEV${dev}_PROTOCOL_MAN DISPLAY_NAME  "Protocol ${dev}"
       set_parameter_property GUI_DEV${dev}_PROTOCOL_MAN DESCRIPTION   \
           "Select the specific type of Protocol used by the memory device on this chip select signal."
       set_parameter_property GUI_DEV${dev}_PROTOCOL_MAN ENABLED        true
       set_parameter_property GUI_DEV${dev}_PROTOCOL_MAN ALLOWED_RANGES \
        {"none" "Generic_x8" "HyperRAM" "HyperFlash" "SemperFlash" "OctaRAM" "OctaFlash" "XccelaRAM" "xSPI_Flash" "xSPI STT-MRAM" "xSPI_PSRAM"}


       # Instantiate Manual Configuration
       #   The Value of the Manual configuration will be loaded into the AUTO configuration as required.
       #
       add_display_item "Memory Channel $dev Configuration"  GUI_DEV${dev}_MODEL  PARAMETER ""

       proc_add_parameter     GUI_DEV${dev}_MODEL String "none" "" false "Memory Channel $dev Configuration"
       set_parameter_property GUI_DEV${dev}_MODEL DISPLAY_NAME  "Memory device on chip select ${dev}"
       set_parameter_property GUI_DEV${dev}_MODEL DESCRIPTION   \
           "Select the specific type of Memory device connected to this chip select signal."
       set_parameter_property GUI_DEV${dev}_MODEL ENABLED        true
          # Create and populate the 'Device X Info' tab.

    #----------------------------------------------------------
    # Change from SPI to DDR mode after Powerup
    #----------------------------------------------------------
    add_display_item "Memory Channel $dev Configuration"   GUI_DEV${dev}_SPI_BOOT     PARAMETER ""

    proc_add_parameter     GUI_DEV${dev}_SPI_BOOT    Integer 0 "" false  "Memory Channel $dev Configuration"
    set_parameter_property GUI_DEV${dev}_SPI_BOOT    DISPLAY_NAME "SPI-DDR Boot Configuraton" ;
    set_parameter_property GUI_DEV${dev}_SPI_BOOT    DESCRIPTION  "Switch from SPI to DDR mode after Power";
    set_parameter_property GUI_DEV${dev}_SPI_BOOT    ALLOWED_RANGES {"1 : Device Powers Up in SPI mode: Switch to DDR mode" \
                                                                     "0 : Device Powers Up in DDR mode: Do nothing" }


   }

  #---------------------------------------------------
  #Add DQ-DQS Skew parameters
  #---------------------------------------------------
  for {set ch 0} {$ch < 4} {incr ch} {
    proc_add_dq_dqs_skew_param $ch
  }

}



#-------------------------------------------------------------------------------
# CLOCK and PLL configuration Tab
#-------------------------------------------------------------------------------
proc proc_add_clock_configuration {} {

add_display_item ""                                     "Clock and PLL Configuration"           GROUP tab

add_display_item "Clock and PLL Configuration"          "Clock Configuration"                   GROUP ""
add_display_item "Clock and PLL Configuration"          "Selected Configuration - Bandwidth"    GROUP ""
add_display_item "Clock and PLL Configuration"          "Selected Configuration - Clocks"       GROUP ""
add_display_item "Clock and PLL Configuration"          "Clock Introspection Report"            GROUP ""
add_display_item "Clock and PLL Configuration"          "About the PLL Integrated In This MBMC Instance"      GROUP ""


add_display_item "Clock Configuration"                  GUI_SINGLE_CLK_OP                   PARAMETER "" ;# Single Clock operation
add_display_item "Clock Configuration"                  GUI_IAVS0_AUTO_CLOCK_RATE           PARAMETER "" ;# MUST BE DISPLAYED UNDER THE GUI_SINGLE_CLK_OP
add_display_item "Clock Configuration"                  GUI_HBUS0_AUTO_CLOCK_RATE           PARAMETER "" ;# MUST BE DISPLAYED UNDER THE GUI_SINGLE_CLK_OP
add_display_item "Clock Configuration"                  GUI_IAVS0_AUTO_CLOCK_DOMAIN          PARAMETER "" ;# MUST BE DISPLAYED UNDER THE GUI_SINGLE_CLK_OP
add_display_item "Clock Configuration"                  GUI_HBUS_AUTO_CLOCK_DOMAIN           PARAMETER "" ;# MUST BE DISPLAYED UNDER THE GUI_SINGLE_CLK_OP
add_display_item "Clock Configuration"                  GUI_UniqueName                       PARAMETER "" ;# MUST BE DISPLAYED UNDER THE GUI_SINGLE_CLK_OP

###==========================================================
### Start Backwards compatability v2.12
###
add_display_item "Clock Configuration"                  MEMORY_FREQ_IN_MHZ                PARAMETER ""  ;# MANUAL VARIABLE
add_display_item "Clock Configuration"                  IAVS_FREQ_OUT_IN_MHZ                PARAMETER ""  ;# MANUAL VARIABLE
###
### End Backwards compatability
###==========================================================

add_display_item "Selected Configuration - Clocks"      GUI_MEMORY_FREQ_IN_MHZ_SC         PARAMETER ""
add_display_item "Selected Configuration - Clocks"      GUI_AV_OUT_FREQ_IN_MHZ_SC           PARAMETER ""




## NO INTEGRATED CLOCK -> CLOCK INTROSPECTION

proc_add_parameter      GUI_IAVS0_AUTO_CLOCK_RATE       LONG 0 hertz true   "Clock Configuration"
set_parameter_property  GUI_IAVS0_AUTO_CLOCK_RATE       DISPLAY_NAME        "i_iavs0_clk frequency is"
set_parameter_property  GUI_IAVS0_AUTO_CLOCK_RATE       VISIBLE             false
set_parameter_property  GUI_IAVS0_AUTO_CLOCK_RATE       SYSTEM_INFO_TYPE    CLOCK_RATE
set_parameter_property  GUI_IAVS0_AUTO_CLOCK_RATE       SYSTEM_INFO_ARG     i_iavs0_clk


proc_add_parameter      GUI_HBUS0_AUTO_CLOCK_RATE       LONG 0 hertz true   "Clock Configuration"
set_parameter_property  GUI_HBUS0_AUTO_CLOCK_RATE       DISPLAY_NAME        "i_hbus_clk_0 frequency is"
set_parameter_property  GUI_HBUS0_AUTO_CLOCK_RATE       VISIBLE             false
set_parameter_property  GUI_HBUS0_AUTO_CLOCK_RATE       SYSTEM_INFO_TYPE    CLOCK_RATE
set_parameter_property  GUI_HBUS0_AUTO_CLOCK_RATE       SYSTEM_INFO_ARG     i_hbus_clk_0

proc_add_parameter      GUI_IAVS0_AUTO_CLOCK_DOMAIN     integer 0 "" true   "Clock Configuration"
set_parameter_property  GUI_IAVS0_AUTO_CLOCK_DOMAIN     DISPLAY_NAME        "i_iavs0_clk domain is"
set_parameter_property  GUI_IAVS0_AUTO_CLOCK_DOMAIN     VISIBLE             false
set_parameter_property  GUI_IAVS0_AUTO_CLOCK_DOMAIN     SYSTEM_INFO_TYPE    CLOCK_DOMAIN
set_parameter_property  GUI_IAVS0_AUTO_CLOCK_DOMAIN     SYSTEM_INFO_ARG     i_iavs0_clk

proc_add_parameter      GUI_HBUS_AUTO_CLOCK_DOMAIN     integer 0 "" true   "Clock Configuration"
set_parameter_property  GUI_HBUS_AUTO_CLOCK_DOMAIN     DISPLAY_NAME        "i_hbus_clk domain is"
set_parameter_property  GUI_HBUS_AUTO_CLOCK_DOMAIN     VISIBLE             false
set_parameter_property  GUI_HBUS_AUTO_CLOCK_DOMAIN     SYSTEM_INFO_TYPE    CLOCK_DOMAIN
set_parameter_property  GUI_HBUS_AUTO_CLOCK_DOMAIN     SYSTEM_INFO_ARG     i_hbus_clk_0

proc_add_parameter      GUI_UniqueName                 STRING "" "" true   "Clock Configuration"
set_parameter_property  GUI_UniqueName                 DISPLAY_NAME        "Unique Name"
set_parameter_property  GUI_UniqueName                 SYSTEM_INFO         UNIQUE_ID

###
# Open Cores only supports  single clock
#
proc_add_parameter     GUI_SINGLE_CLK_OP BOOLEAN false "" false  "Clock Configuration"
set_parameter_property GUI_SINGLE_CLK_OP ALLOWED_RANGES {"true:One clock-One clock for all the Avalon ports and the Memory channel" \
                                                         "false:Two clocks-One clock for all the Avalon ports. One clock for the Memory channel" }

set_parameter_property GUI_SINGLE_CLK_OP DISPLAY_NAME  "Avalon and Memory clock configuration"
set_parameter_property GUI_SINGLE_CLK_OP DESCRIPTION   "\
    The 'One clock for all the Avalon ports and the Memory channel' configuration has the lowest circuit area and is well suited for severely resource constrained designs.\
    The 'One clock for all the Avalon ports. One clock for the Memory channel' configuration employs a dual clock configuration. \
    In this configuration, SLL MBMC implements low-area, high-bandwidth, clock-crossing logic between the Avalon bus target and Memory channel."

##


# This variable is used to set the HyperBus frequency.
#
proc_add_parameter      MEMORY_FREQ_IN_MHZ   INTEGER   150   megahertz   false   "Clock Configuration"
set_parameter_property  MEMORY_FREQ_IN_MHZ   DISPLAY_NAME                        "Memory channel clock frequency"
set_parameter_property  MEMORY_FREQ_IN_MHZ   DESCRIPTION    "\
  This is the clock frequency used to drive the memory channel.\
  The wire-speed bandwidth of the 8-bit wide Double Data Rate (DDR) memory channel is: 2xMHz bytes per second."

# THIS VARIABLE SETS THE AVALON SHARED BUS SPEED, AND CAN SOMETIMES SET THE MEMORY FREQUENCY
#
proc_add_parameter      IAVS_FREQ_OUT_IN_MHZ   INTEGER   100   megahertz   false   "Clock Configuration"
set_parameter_property  IAVS_FREQ_OUT_IN_MHZ   DISPLAY_NAME                        "Shared Avalon clock frequency"
set_parameter_property  IAVS_FREQ_OUT_IN_MHZ   DESCRIPTION                         "\
  This is the clock frequency used to drive all the Avalon ports."


###
### End Backwards compatability
###==========================================================


## CLOCKS REPORT
#
proc_add_parameter     GUI_AV_OUT_FREQ_IN_MHZ_SC INTEGER 100 megahertz true "Selected Configuration - Clocks"
set_parameter_property GUI_AV_OUT_FREQ_IN_MHZ_SC DISPLAY_NAME   "Avalon output clock (av_out_clk) frequency"
set_parameter_property GUI_AV_OUT_FREQ_IN_MHZ_SC VISIBLE        true
set_parameter_property GUI_AV_OUT_FREQ_IN_MHZ_SC ENABLED            false
set_parameter_property GUI_AV_OUT_FREQ_IN_MHZ_SC DESCRIPTION        "\
    This is the clock frequency that the PLL in this MBMC instance generates for the clock source o_av_out_clk.\
    The o_av_out_clk is used to drive the avalon bus. It must also be connected to the i_iavs0_clk clock sink."


proc_add_parameter     GUI_MEMORY_FREQ_IN_MHZ_SC INTEGER 100 megahertz true "Selected Configuration - Clocks"
set_parameter_property GUI_MEMORY_FREQ_IN_MHZ_SC DISPLAY_NAME     "Memory channel clock frequency"
set_parameter_property GUI_MEMORY_FREQ_IN_MHZ_SC VISIBLE          true
set_parameter_property GUI_MEMORY_FREQ_IN_MHZ_SC ENABLED          false
set_parameter_property GUI_MEMORY_FREQ_IN_MHZ_SC DESCRIPTION      "\
    This is the clock frequency used to drive just the memory channel.\
    The peak sustainable bandwidth of the 8-bit wide Double Data Rate (DDR)\
    memory channel is: 2xMHz bytes per second."

}


#-------------------------------------------------------------------------------
#set display group
#-------------------------------------------------------------------------------
proc proc_validate_dev_cfgs {} {
    # variable needed to check data width
    set v_dev_xdq_width_selected   0

    #
    #get number of supported memory devices
    #
    set v_num_chipselect           [get_parameter_value GUI_NUM_CHIPSELECTS]

    #
    #get Memory channel frequency
    #
    set v_memory_frequency_mhz     [get_parameter_value GUI_MEMORY_FREQ_IN_MHZ_SC]


    #++++++++++++++++++++++++++++++
    #burst count
    #++++++++++++++++++++++++++++++
    set v_iavs0_gui_burstcount [ get_parameter_value GUI_IAVS0_MAX_BURST_IN_WORDS]


    #---------------------------------------------------------------
    #check configuration for generic x8
    #---------------------------------------------------------------
    set v_dev0_protocol         [get_parameter_value GUI_DEV0_PROTOCOL_MAN]
    set v_dev1_protocol         [get_parameter_value GUI_DEV1_PROTOCOL_MAN]

    if {$v_dev0_protocol=="Generic_x8" && $v_dev1_protocol != "none" } {
        send_message error  "For Generic x8 mode, plase select  \
                              Generic_x8 on Memory Channel 0 and \
                              None       on Memory Channel 1"
    }

    if {$v_dev1_protocol=="Generic_x8" } {
        send_message error  "This option is not supported on Memory Channel 1 . \
                             Please select another option."
    }


    #++++++++++++++++++++++++++++++
    # check whether Register port is enabled generic_x8 is set
    #++++++++++++++++++++++++++++++
    if {($v_dev0_protocol == "Generic_x8") && ([get_parameter_value GUI_INCLUDE_REG_AVALON] == 0)} {
        send_message error  "Register Avalon Port needs to be enabled for x8 generic support. \
                            Please enable the  Register Avalon Port in the Master Configuration Tab"
    }

    #---------------------------------------------------------------
    #check configuration
    #---------------------------------------------------------------
    for {set dev 0} {$dev < 2} {incr dev} {
        #
        # Do we check this device tab?
        #
        if {$dev <  $v_num_chipselect } {
           set v_dev_name         [get_parameter_value GUI_DEV${dev}_NAME]
           set v_dev_max_speed    [get_parameter_value GUI_DEV${dev}_SPEED ]
           set v_dev_is_flash     [get_parameter_value GUI_DEV${dev}_IS_FLASH]
           set v_dev_is_xspi      [get_parameter_value GUI_DEV${dev}_IS_XSPI]
           set v_dev_tacc         [get_parameter_value GUI_DEV${dev}_T_ACC]
           set v_dev_tcss         [get_parameter_value GUI_DEV${dev}_T_CSS]
           set v_dev_tcsh         [get_parameter_value GUI_DEV${dev}_T_CSH]
           set v_dev_xdq_width    [get_parameter_value GUI_DEV${dev}_DQ_WIDTH]

           if {$v_dev_name != "none"} {
              #++++++++++++++++++++++++++++++
              #check whether we have different memory data widths
              #Only devices with same memory data widths are supported
              #++++++++++++++++++++++++++++++

              if {$v_dev_xdq_width_selected == 0} {
                 set v_dev_xdq_width_selected   $v_dev_xdq_width
              } else {
                 if {$v_dev_xdq_width_selected != $v_dev_xdq_width} {
                    send_message error  "Memories on Chip select 0, 1 have different I/O Data width.  \
                                         This is not supported. Please ensure that both memories have \
                                         the same  I/O Data width."
                 }
              }

              #++++++++++++++++++++++++++++++
              #Check Chip select Low (4us / 1us)
              #++++++++++++++++++++++++++++++

              #for Hyperflash, we do not double the Init latency time
              if {$v_dev_is_flash == 1} {
                set v_max_ce_low_time   [expr  3 + ($v_iavs0_gui_burstcount*2) + (  $v_dev_tacc) + $v_dev_tcss + $v_dev_tcsh]
              } else {
                set v_max_ce_low_time   [expr  3 + ($v_iavs0_gui_burstcount*2) + (2*$v_dev_tacc) + $v_dev_tcss + $v_dev_tcsh]
              }

              set vendor_tCE     4000
              set vendor_tCE_ind 1000

              #convert to ns
              set v_max_ce_low_time     [expr ($v_max_ce_low_time*1000) /$v_memory_frequency_mhz]

              if {($v_dev_is_flash == 0) && ($v_max_ce_low_time > $vendor_tCE_ind)} {
                 send_message Warning  "Total Chip Select Low (Tcsm) time is greater than $vendor_tCE_ind ns. \
                                        The range is acceptable for Industrial Design projects,  \
                                        For Automotive/Extended Temperature Design projects,  \
                                        please reduce Max Burst Size in the Avalon (IAVS0) Configuration Tab"
              }

              if {($v_dev_is_flash == 0) && ($v_max_ce_low_time > $vendor_tCE)} {
                  send_message error  "Total Chip Select Low (Tcsm) time is greater than $vendor_tCE ns. \
                                      Please reduce Max Burst Size in the Avalon (IAVS0) Configuration Tab"
              }


              #++++++++++++++++++++++++++++++
              #For xSPI model, Register port needs to be enabled
              #++++++++++++++++++++++++++++++
              if {($v_dev_is_xspi == 1) && ([get_parameter_value GUI_INCLUDE_REG_AVALON] == 0)} {
                  send_message error  "Register Avalon Port needs to be enabled for this Device. \
                                      Please enable the  Register Avalon Port in the Master Configuration Tab"
              }


              #Check whether selected speed is supported on this device
              if {$v_dev_name != "none" && ($v_memory_frequency_mhz > $v_dev_max_speed) } {
                send_message error  "Current selected Frequency ($v_memory_frequency_mhz) is not supported for this device. \
                                    Please reduce the memory channel frequency."
              }
           }

        }
    }


    #check FPGA compatibility
    set v_device_family [ get_parameter_value DEVICE_FAMILY ]

    if { $v_device_family == "Arria 10" ||
           $v_device_family == "ARRIA 10" ||
           $v_device_family == "Cyclone 10 GX" ||
           $v_device_family == "CYCLONE 10 GX" ||
           $v_device_family == "STRATIX 10" ||
           $v_device_family == "Stratix 10"} {

       if { [get_parameter_value GUI_PYHLITE_INTERFACE]} {
       } else {
           send_message Warning "In this configuration, High Clock speeds are not supported on the $v_device_family FPGA family.  \
                               If you have an R&D or Production license, please select PHYLITE interface otherwise \
                               send an email to info@synatic-labs.com for more info "
       }


    } else {
       if { [get_parameter_value GUI_PYHLITE_INTERFACE] } {
           send_message Error "PHYlite interface not supported on $v_device_family \
                               PHYlite interface only supported by ARRIA 10, Cyclone 10 GX and STRATIX 10 FPGA family. \
                               Please change the Select I/O Interface Parameter in the Master Configuration Tab to Standard I/O Interface"
       }
    }


    #
    #check FPGA compatibility for Cyclone V
    #
    if { $v_device_family == "Cyclone V" ||
         $v_device_family == "CYCLONE V" } {


       if { [get_parameter_value GUI_DQIN_MODE] == 1} {
          send_message Info "Ensure to add set_global_assignment -name VERILOG_MACRO CYCLONE_V_DDIO=1 in QSF file"
       } else {
          send_message Info "Ensure to add set_global_assignment -name VERILOG_MACRO CYCLONE_V=1 in QSF file"
       }


   }

    #check FPGA compatibility for MAX 10
 ##    if { $v_device_family == "Max 10" ||
 ##         $v_device_family == "MAX 10" } {
 ##       if { [get_parameter_value GUI_DQIN_MODE] == 1} {
 ##        send_message Info "Some MAX 10 devices support IO PADS with integrated DDR IN registers. \
 ##                           To enable this functionality, add the following in QSF file : \
 ##                           set_global_assignment -name VERILOG_MACRO SLL_USE_ADVANCED_DDR_FEATURES_MAX10=1 "
 ##     }
 ##   }
    #Setting data width
    set v_iavs0_gui_data_width  [ get_parameter_value GUI_IAVS0_DATA_WIDTH]

    #++++++++++++++++++++++++++++++
    # Check the word width of the IAVS memory access port.
    #+++++++++++++++
    #
    if {$v_iavs0_gui_data_width != 32} {
        send_message error  "\
            The word width of IAVS0 is currently set to $v_iavs0_gui_data_width.\
            In this case, this value should be set to 32."
    }



    #
    # END ERROR MESSAGES THAT REQUIRE MANUAL INTERVENTION ON UPGRADE BETWEEN VERSIONS
    #
    #####################################################################################
}


# +-----------------------------------
# |
# | Callback function when verifying whether a given parameter is
# | valid on this release.
# |
# +-----------------------------------
##set_module_property PARAMETER_UPGRADE_CALLBACK my_parameter_upgrade
##
##proc my_parameter_upgrade {ip_core_type version parameters} {
##  send_message Error "XSPI_MC Version Upgrade - Manual Intervention Required: \
##                      For Dual Clock operation, \
##                      Ensure you connect the Avalon clock to the third port on the PLL ( instead of the fourth one)."
##
##}

#-------------------------------------------------------------------------------
#set display group
#-------------------------------------------------------------------------------
proc proc_set_display_group {NAME GROUP EXPERIMENTAL DISPLAY_NAME args} {
    add_display_item        $GROUP  $NAME               parameter
    set_parameter_property  $NAME   "DISPLAY_NAME"      "$DISPLAY_NAME"
    set display_message     "$args"
    # only show those settings in debug mode
    if { "$EXPERIMENTAL" == "1" } {
        set_parameter_property  $NAME   "STATUS"        "EXPERIMENTAL"
        set_parameter_property  $NAME   "VISIBLE"       false
    } else {
        set_parameter_property  $NAME   "VISIBLE"       true
    }

    if { [ expr { "DES_$args" != "DES_" } ] } {
        set_parameter_property  $NAME   "DESCRIPTION"   "[ join $display_message ]"
    }
}


#-------------------------------------------------------------------------------
#function : add parameter
#-------------------------------------------------------------------------------
proc proc_add_parameter {NAME TYPE DEFAULT UNITS IS_DERIVED GROUP} {
    add_parameter          $NAME  $TYPE
    set_parameter_property $NAME AFFECTS_ELABORATION    true                       ;# Set AFFECTS_ELABORATION to false for parameters that do not affect the external interface of the module.
    set_parameter_property $NAME AFFECTS_VALIDATION     true                       ;# The AFFECTS_VALIDATION property marks whether a parameter's value is used to set derived parameters, and whether the value affects validation messages.
    set_parameter_property $NAME AFFECTS_GENERATION     true                       ;# Set AFFECTS_GENERATION to false if the value of a parameter does not change the results of fileset generation.
    set_parameter_property $NAME HDL_PARAMETER          false                      ;# When true, the parameter must be passed to the HDL IP component description. The default value is false.
    set_parameter_property $NAME DERIVED                $IS_DERIVED                ;# When true, indicates that the parameter value can only be set by the IP component, and cannot be set by the user.
    set_parameter_property $NAME ENABLED                true                       ;# When false, the parameter is disabled, meaning that it is displayed, but greyed out, indicating that it is not editable on the parameter editor.
    set_parameter_property $NAME DEFAULT_VALUE          $DEFAULT
    set_parameter_property $NAME NEW_INSTANCE_VALUE     $DEFAULT
    set_parameter_property $NAME UNITS                  $UNITS                     ;# This is the GUI label that appears to the right of the parameter.
}


#-------------------------------------------------------------------------------
#function : add hdl parameter
#-------------------------------------------------------------------------------
proc proc_add_hdl_parameter {NAME TYPE DEFAULT} {
    add_parameter          $NAME  $TYPE
    set_parameter_property $NAME  AFFECTS_GENERATION    false                      ;# Set AFFECTS_GENERATION to false if the value of a parameter does not change the results of fileset generation.
    set_parameter_property $NAME  AFFECTS_ELABORATION   false                      ;# Set AFFECTS_ELABORATION to false for parameters that do not affect the external interface of the module.
    set_parameter_property $NAME  AFFECTS_VALIDATION    false                      ;# The AFFECTS_VALIDATION property marks whether a parameter's value is used to set derived parameters, and whether the value affects validation messages.
    set_parameter_property $NAME  HDL_PARAMETER         true                       ;# When true, the parameter must be passed to the HDL IP component description. The default value is false.
    set_parameter_property $NAME  DERIVED               true                       ;# When true, indicates that the parameter value can only be set by the IP component, and cannot be set by the user.
    set_parameter_property $NAME  DEFAULT_VALUE         $DEFAULT
    set_parameter_property $NAME  NEW_INSTANCE_VALUE    $DEFAULT
    set_parameter_property $NAME  VISIBLE               true                       ;# Indicates whether or not to display the parameter in the parameterization GUI.
    set_parameter_property $NAME  ENABLED               false                      ;# When false, the parameter is disabled, meaning that it is displayed, but greyed out, indicating that it is not editable on the parameter editor.
    set_parameter_property $NAME  GROUP                 "Derived Parameters"       ;# Controls the layout of parameters in GUI
    set_parameter_property $NAME  DISPLAY_NAME          $NAME
}

#-------------------------------------------------------------------------------
#function : add avalon slave
#-------------------------------------------------------------------------------

proc proc_add_avalon_slave {NAME IS_MEM} {
    add_interface             $NAME  avalon                slave
    set_interface_property    $NAME  addressUnits          WORDS
    set_interface_property    $NAME  bitsPerSymbol         8
    set_interface_property    $NAME  bridgedAddressOffset  0
    set_interface_property    $NAME  burstcountUnits       WORDS
    set_interface_property    $NAME  alwaysBurstMaxBurst   {0}
    set_interface_property    $NAME  constantBurstBehavior {0}
    set_interface_property    $NAME  holdTime              {0}
    set_interface_property    $NAME  setupTime             {0}
    set_interface_property    $NAME  writeLatency          {0}
    set_interface_property    $NAME  explicitAddressSpan   0
    set_interface_property    $NAME  timingUnits           Cycles
    set_interface_property    $NAME  isMemoryDevice        $IS_MEM

    set_interface_property    $NAME  EXPORT_OF             ""
    set_interface_property    $NAME  PORT_NAME_MAP         ""
    set_interface_property    $NAME  CMSIS_SVD_VARIABLES   ""
    set_interface_property    $NAME  SVD_ADDRESS_GROUP     ""
}

#-------------------------------------------------------------------------------
#function : add avalon Register slave
#-------------------------------------------------------------------------------
proc proc_add_reg_avalon_slave {NAME RD_WAIT } {
   set_interface_property    $NAME  addressUnits                      WORDS
   set_interface_property    $NAME  bitsPerSymbol                     8
   set_interface_property    $NAME  bridgedAddressOffset              0
   set_interface_property    $NAME  bridgesToMaster                   ""
##   set_interface_property    $NAME  burstOnBurstBoundariesOnly        true  ; # This is typically true, unless we have special burst support logic in our slave that supports non-aligned bursts
   set_interface_property    $NAME  burstcountUnits                   WORDS
   set_interface_property    $NAME  alwaysBurstMaxBurst               {0}
   set_interface_property    $NAME  constantBurstBehavior             {0}  ; # This should be true for our IP and implementation. Our IP should not rely on the burst-length being held throughout a write burst transaction.
   set_interface_property    $NAME  holdTime                          {0}
   set_interface_property    $NAME  readLatency                       {1}  ; # This value is always zero for pipelined interfaces.
   set_interface_property    $NAME  readWaitTime                      {0}
   set_interface_property    $NAME  setupTime                         {0}
   set_interface_property    $NAME  writeLatency                      {0}
   set_interface_property    $NAME  writeWaitStates                   {0}
   set_interface_property    $NAME  writeWaitTime                     {0}
   set_interface_property    $NAME  maximumPendingReadTransactions    {0}  ; # This value is always 1, unless we have an advanced IP that is literally processing 2 or more read transactions concurrently within its pipeline.
   set_interface_property    $NAME  maximumPendingWriteTransactions   {0}    ; # This value is only used with Avalon Response codes to Write operations.  Typically, this value is 0 in our IP.
   set_interface_property    $NAME  explicitAddressSpan               0
   set_interface_property    $NAME  linewrapBursts                    false  ; # This value is typically zero for our IP.
   set_interface_property    $NAME  timingUnits                       Cycles

   add_interface_port        $NAME  i_iavsr_addr            address         Input   6
   add_interface_port        $NAME  i_iavsr_do_wr           write           Input   1
   add_interface_port        $NAME  i_iavsr_do_rd           read            Input   1
   add_interface_port        $NAME  i_iavsr_wdata           writedata       Input   32
   add_interface_port        $NAME  o_iavsr_rdata           readdata        Output  32

   set_interface_assignment  $NAME  embeddedsw.configuration.isFlash              0
   set_interface_assignment  $NAME  embeddedsw.configuration.isMemoryDevice       0
   set_interface_assignment  $NAME  embeddedsw.configuration.isNonVolatileStorage 0
   set_interface_assignment  $NAME  embeddedsw.configuration.isPrintableDevice    0
}

#-------------------------------------------------------------------------------
#function : add avalon Register slave
#-------------------------------------------------------------------------------
proc proc_add_reg_apb_slave {NAME } {

   set_interface_property    $NAME ENABLED true
   set_interface_property    $NAME EXPORT_OF ""
   set_interface_property    $NAME PORT_NAME_MAP ""
   set_interface_property    $NAME CMSIS_SVD_VARIABLES ""
   set_interface_property    $NAME SVD_ADDRESS_GROUP ""

   add_interface_port        $NAME  apb0_paddr            paddr         Input   8
   add_interface_port        $NAME  apb0_psel             psel          Input   1
   add_interface_port        $NAME  apb0_penable          penable       Input   1
   add_interface_port        $NAME  apb0_pwrite           pwrite        Input   1
   add_interface_port        $NAME  apb0_pwdata           pwdata        Input   32
   add_interface_port        $NAME  apb0_prdata           prdata        Output  32
   add_interface_port        $NAME  apb0_pready           pready        Output  1
   add_interface_port        $NAME  apb0_pslverr          pslverr       Output  1

   set_interface_assignment  $NAME  embeddedsw.configuration.isFlash              0
   set_interface_assignment  $NAME  embeddedsw.configuration.isMemoryDevice       0
   set_interface_assignment  $NAME  embeddedsw.configuration.isNonVolatileStorage 0
   set_interface_assignment  $NAME  embeddedsw.configuration.isPrintableDevice    0
}

# +-------------------------------------
# | Add settings needed by Nios tools
# +-------------------------------------
proc proc_gen_embeddedsw_params {NAME dev } {

    # Tells us component is a flash but not a EPCS device
    set_module_assignment embeddedsw.memoryInfo.IS_FLASH         1

    # This assignment indicates Nios tools should create a Flash initialization file
    set_module_assignment embeddedsw.memoryInfo.HAS_BYTE_LANE    0

    # These assignments tells tools to create byte_addressed .hex (syntehsis) and .dat (simulation) files
    set_module_assignment embeddedsw.memoryInfo.GENERATE_HEX     1
    set_module_assignment embeddedsw.memoryInfo.USE_BYTE_ADDRESSING_FOR_HEX 1
    set_module_assignment embeddedsw.memoryInfo.GENERATE_DAT_SYM 1
    set_module_assignment embeddedsw.memoryInfo.GENERATE_FLASH   1

    # Width of memory
    set_module_assignment embeddedsw.memoryInfo.MEM_INIT_DATA_WIDTH 32

    # Output directories for programming files
    set_module_assignment embeddedsw.memoryInfo.DAT_SYM_INSTALL_DIR {SIM_DIR}
    set_module_assignment embeddedsw.memoryInfo.FLASH_INSTALL_DIR   {APP_DIR}
    set_module_assignment embeddedsw.memoryInfo.HEX_INSTALL_DIR     {QPF_DIR}

    # Interface assignments that indicate the slave is a non-volatile flash memory

    # Module assignments related to names of simulation files
    #set_module_assignment postgeneration.simulation.init_file.param_name {INIT_FILENAME}
    #set_module_assignment postgeneration.simulation.init_file.type {MEM_INIT}

    # interface assignments for embedded software
    set_interface_assignment $NAME embeddedsw.configuration.isFlash              1
    set_interface_assignment $NAME embeddedsw.configuration.isMemoryDevice       1
    set_interface_assignment $NAME embeddedsw.configuration.isNonVolatileStorage 1
    set_interface_assignment $NAME embeddedsw.configuration.isPrintableDevice    0


    #get device name
    set v_dev_size              [get_parameter_value GUI_DEV${dev}_SIZE]
    set v_dev_size_in_bytes     [expr  ($v_dev_size *1024*1024)]
    set v_memory_frequency_mhz  [get_parameter_value GUI_MEMORY_FREQ_IN_MHZ_SC]

    set_module_assignment embeddedsw.CMacro.SIZE              $v_dev_size_in_bytes
    set_module_assignment embeddedsw.CMacro.MEM_FREQ_MHZ      $v_memory_frequency_mhz


}

#-------------------------------------------------------------------------------
#Add skew parameters
#-------------------------------------------------------------------------------
proc proc_add_dq_dqs_skew_param {dev } {

add_display_item "Memory RWDS-DQ SKEW Configuration"  GUI_CH${dev}_DQ_DQS_SKEW     PARAMETER "" ;# FPGA board
add_display_item "Memory RWDS-DQ SKEW Configuration"  GUI_CH${dev}_DQ_DQS_SKEW_MAN PARAMETER "" ;# FPGA board


proc_add_parameter          GUI_CH${dev}_DQ_DQS_SKEW          integer 800 picoseconds true "Memory RWDS-DQ SKEW Configuration"
set_parameter_property      GUI_CH${dev}_DQ_DQS_SKEW          DISPLAY_NAME  "DQ-DQS skew on channel ${dev} in ps"
set_parameter_property      GUI_CH${dev}_DQ_DQS_SKEW          DESCRIPTION   "DQ-DQS skew on channel ${dev} in ps"
set_parameter_property      GUI_CH${dev}_DQ_DQS_SKEW          ENABLED        true
set_parameter_property      GUI_CH${dev}_DQ_DQS_SKEW          VISIBLE        true

proc_add_parameter          GUI_CH${dev}_DQ_DQS_SKEW_MAN      integer 800 picoseconds false "Memory RWDS-DQ SKEW Configuration"
set_parameter_property      GUI_CH${dev}_DQ_DQS_SKEW_MAN      DISPLAY_NAME  "M DQ-DQS skew on channel ${dev} in ps"
set_parameter_property      GUI_CH${dev}_DQ_DQS_SKEW_MAN      DESCRIPTION   "M DQ-DQS skew on channel ${dev} in ps"
set_parameter_property      GUI_CH${dev}_DQ_DQS_SKEW_MAN      ENABLED        true
set_parameter_property      GUI_CH${dev}_DQ_DQS_SKEW_MAN      VISIBLE        false


}

#-------------------------------------------------------------------------------
#procedure : proc_add_avalon_conduit
#-------------------------------------------------------------------------------

proc proc_add_avalon_conduit {NAME SLAVE_TYPE PendReads} {

        add_interface_port     $NAME  i_${NAME}_addr            address         Input   g_${NAME}_addr_width
        add_interface_port     $NAME  i_${NAME}_burstcount      burstcount      Input   g_${NAME}_burstcount_width
        add_interface_port     $NAME  o_${NAME}_wait_request    waitrequest     Output  1


    if {$SLAVE_TYPE != "AV_READ_ONLY"} {
        add_interface_port     $NAME  i_${NAME}_do_wr           write           Input   1
        add_interface_port     $NAME  i_${NAME}_byteenable      byteenable      Input   g_${NAME}_av_numsymbols
        add_interface_port     $NAME  i_${NAME}_wdata           writedata       Input   g_${NAME}_data_width

        set_interface_property $NAME  writeWaitStates                    {0}
        set_interface_property $NAME  writeWaitTime                      {0}
        set_interface_property $NAME  maximumPendingWriteTransactions    {0}    ; # This value is only used with Avalon Response codes to Write operations.  Typically, this value is 0 in our IP.
    }

    if {$SLAVE_TYPE != "AV_WRITE_ONLY"} {
        add_interface_port     $NAME  i_${NAME}_do_rd           read            Input   1
        add_interface_port     $NAME  o_${NAME}_rdata           readdata        Output  g_${NAME}_data_width
        add_interface_port     $NAME  o_${NAME}_rdata_valid     readdatavalid   Output  1
        add_interface_port     $NAME  o_${NAME}_resp            response        Output  2

        set_interface_property $NAME  maximumPendingReadTransactions     $PendReads  ; # This value is always 1, unless we have an advanced IP that is literally processing 2 or more read transactions concurrently within its pipeline.
        set_interface_property $NAME  readLatency                        {0}  ; # This value is always zero for pipelined interfaces.
        set_interface_property $NAME  readWaitTime                       {0}
    }
}

#-------------------------------------------------------------------------------
#function : Device information Menu
#-------------------------------------------------------------------------------
proc proc_add_dev_info {dev  } {

    add_display_item "" "Device ${dev} Info" GROUP tab

    add_display_item "Device ${dev} Info" "Device ${dev} Parameters"  GROUP ""
    add_display_item "Device ${dev} Info" "Device ${dev} Timings"     GROUP ""

    add_display_item "Device ${dev} Parameters"      GUI_DEV${dev}_NAME         PARAMETER ""
    add_display_item "Device ${dev} Parameters"      GUI_DEV${dev}_TYPE         PARAMETER ""
    add_display_item "Device ${dev} Parameters"      GUI_DEV${dev}_IS_FLASH     PARAMETER ""
    add_display_item "Device ${dev} Parameters"      GUI_DEV${dev}_IS_XSPI      PARAMETER ""
    add_display_item "Device ${dev} Parameters"      GUI_DEV${dev}_SIZE         PARAMETER ""
    add_display_item "Device ${dev} Parameters"      GUI_DEV${dev}_SPEED        PARAMETER ""
    add_display_item "Device ${dev} Parameters"      GUI_DEV${dev}_DQ_WIDTH     PARAMETER ""
    add_display_item "Device ${dev} Parameters"      GUI_DEV${dev}_WRAP_SUPPORT PARAMETER ""
    add_display_item "Device ${dev} Parameters"      GUI_DEV${dev}_NON_DEFAULT  PARAMETER ""


    add_display_item "Device ${dev} Timings"         GUI_DEV${dev}_T_ACC        PARAMETER ""
    add_display_item "Device ${dev} Timings"         GUI_DEV${dev}_T_RWR        PARAMETER ""
    add_display_item "Device ${dev} Timings"         GUI_DEV${dev}_T_CSS        PARAMETER ""
    add_display_item "Device ${dev} Timings"         GUI_DEV${dev}_T_CSH        PARAMETER ""


    #----------------------------------------------------------
    #Memory device info
    #----------------------------------------------------------
    proc_add_parameter     GUI_DEV${dev}_NAME         String "No device configured" "" true  "Device ${dev} Parameters"
    set_parameter_property GUI_DEV${dev}_NAME         DISPLAY_NAME "Device" ;
    set_parameter_property GUI_DEV${dev}_NAME         DESCRIPTION  \
        "The current configuration of the Memory Memory controller for this chip select.\
        A Memory device may be physically connected to this chip select, but not configured by the XSPI_MC instance.\
        In which case the XSPI_MC will never issue memory transfer requests to the Memory device connected to this chip select."

    proc_add_parameter     GUI_DEV${dev}_SIZE         Integer 0 Megabytes true  "Device ${dev} Parameters"
    set_parameter_property GUI_DEV${dev}_SIZE         DISPLAY_NAME "Device storage capacity" ;
    set_parameter_property GUI_DEV${dev}_SIZE         DESCRIPTION  "Storage capacity of the configured device"

    proc_add_parameter     GUI_DEV${dev}_DQ_WIDTH     Integer 0 "" true  "Device ${dev} Parameters"
    set_parameter_property GUI_DEV${dev}_DQ_WIDTH     DISPLAY_NAME "Device DQ data width" ;
    set_parameter_property GUI_DEV${dev}_DQ_WIDTH     DESCRIPTION  "Device DQ data width"

    proc_add_parameter     GUI_DEV${dev}_SPEED        Integer 0 Megahertz true  "Device ${dev} Parameters"
    set_parameter_property GUI_DEV${dev}_SPEED        DISPLAY_NAME "Device Speed Grade" ;
    set_parameter_property GUI_DEV${dev}_SPEED        DESCRIPTION  "Maximum Device Speed supported by the configured device"

    proc_add_parameter     GUI_DEV${dev}_TYPE         String  "None" "" true  "Device ${dev} Parameters"
    set_parameter_property GUI_DEV${dev}_TYPE         DISPLAY_NAME "Memory Type" ;
    set_parameter_property GUI_DEV${dev}_TYPE         DESCRIPTION  "Memory Type"

    proc_add_parameter     GUI_DEV${dev}_WRAP_SUPPORT Integer 0 "" true  "Device ${dev} Parameters"
    set_parameter_property GUI_DEV${dev}_WRAP_SUPPORT DISPLAY_NAME "Device Wrap Support" ;
    set_parameter_property GUI_DEV${dev}_WRAP_SUPPORT DESCRIPTION  "Device Wrap Support"


    proc_add_parameter     GUI_DEV${dev}_IS_FLASH  Integer 0 "" true  "Device ${dev} Configuration"
    set_parameter_property GUI_DEV${dev}_IS_FLASH  DISPLAY_NAME "Is this Nonvolatile Flash ?" ;
    set_parameter_property GUI_DEV${dev}_IS_FLASH  VISIBLE          false ;

    proc_add_parameter     GUI_DEV${dev}_IS_XSPI   Integer 0 "" true  "Device ${dev} Configuration"
    set_parameter_property GUI_DEV${dev}_IS_XSPI   DISPLAY_NAME "Is this xSPI device ?" ;
    set_parameter_property GUI_DEV${dev}_IS_XSPI   VISIBLE      false ;


    proc_add_parameter     GUI_DEV${dev}_NON_DEFAULT  boolean true "" true  "Device ${dev} Configuration"
    set_parameter_property GUI_DEV${dev}_NON_DEFAULT  DISPLAY_NAME "Non-default configuration settings are programmed after Power-Up" ;

    #----------------------------------------------------------
    #Memory device timing
    #----------------------------------------------------------

    add_display_item "Device ${dev} Parameters"  GUI_DEV${dev} PARAMETER ""

    proc_add_parameter     GUI_DEV${dev}_T_ACC          Integer 0 Cycles true  "Device ${dev} Timings"
    set_parameter_property GUI_DEV${dev}_T_ACC          DISPLAY_NAME "Tacc" ;
    set_parameter_property GUI_DEV${dev}_T_ACC          DESCRIPTION  "Initial Latency. Please refer to the Memory specifications for more details."

    proc_add_parameter     GUI_DEV${dev}_T_RWR          Integer 0 Cycles true  "Device ${dev} Timings"
    set_parameter_property GUI_DEV${dev}_T_RWR          DISPLAY_NAME "Tcshi - Trwr" ;
    set_parameter_property GUI_DEV${dev}_T_RWR          DESCRIPTION  "Read write recovery. Please refer to the Memory specifications for more details."

    proc_add_parameter     GUI_DEV${dev}_T_CSS          Integer 0 Cycles true  "Device ${dev} Timings"
    set_parameter_property GUI_DEV${dev}_T_CSS          DISPLAY_NAME "Tcss" ;
    set_parameter_property GUI_DEV${dev}_T_CSS          DESCRIPTION  "Tcss. Please refer to the Memory specifications for more details."

    proc_add_parameter     GUI_DEV${dev}_T_CSH          Integer 0 Cycles true  "Device ${dev} Timings"
    set_parameter_property GUI_DEV${dev}_T_CSH          DISPLAY_NAME "Tcsh" ;
    set_parameter_property GUI_DEV${dev}_T_CSH          DESCRIPTION  "Tcsh. Please refer to the Memory specifications for more details."

}


# -----------------------------------------------------------------
# Get parameters for
# -----------------------------------------------------------------

proc proc_get_device_parametes {dev0_model dev1_model} {

   # -----------------------------------------------------------------
   #parameter file
   # -----------------------------------------------------------------
   set dev_param_file "sll_ca_mbmc_mem_devices.txt"

   # -----------------------------------------------------------------
   #Read file with parameters
   # -----------------------------------------------------------------
   set v_dev0_found    0
   set v_dev1_found    0

   set dev_file_ptr   [open  ${dev_param_file} r]

   while {[gets $dev_file_ptr line]!=-1} {
       # To make sure that the line has 17 words
       if {[llength $line]!=20} {
           send_message Error "Incorrect format in the line -> '$line'"
           continue
       }
       set v_model_from_file [lindex $line 0]

       #set parameters for device 0
       if {$v_model_from_file == $dev0_model} {
         set v_dev0_list  $line
         set v_dev0_found 1
       }

       #set parameters for device 1
       if {$v_model_from_file == $dev1_model} {
         set v_dev1_list $line
         set v_dev1_found 1
       }

   }

   close $dev_file_ptr

   # -----------------------------------------------------------------
   #check whether device is found
   # -----------------------------------------------------------------
   if {$v_dev0_found == 0 } {
     send_message Error "Device 0 not found : $dev0_model "
   }

   if {$v_dev1_found == 0 } {
     send_message Error "Device 1 not found : $dev1_model "
   }

   return [list $v_dev0_list $v_dev1_list]

}



#-------------------------------------------------------------------------------
#function : Device Timing Menu
#-------------------------------------------------------------------------------

proc proc_set_device_info {dev v_dev_parameters_list} {

    # -----------------------------------------------------------------
    # check whether Genric Profile is enables
    # -----------------------------------------------------------------
    if { [get_parameter_value GUI_DEV${dev}_PROTOCOL_MAN] == "Generic_x8"} {
      set v_dev_is_generic                      1
      set_parameter_value g_reg_generic_mode    1
    } else {
      set v_dev_is_generic 0
    }

    # -----------------------------------------------------------------
    # Get Device 0 parameters
    # -----------------------------------------------------------------
    set v_dev_name             [lindex $v_dev_parameters_list 0]
    set v_dev_type             [lindex $v_dev_parameters_list 1]
    set v_dev_voltage          [lindex $v_dev_parameters_list 2]
    set v_dev_speed            [lindex $v_dev_parameters_list 3]
    set v_dev_size             [lindex $v_dev_parameters_list 4]
    set v_dev_dq_width         [lindex $v_dev_parameters_list 5]
    set v_dev_config         0x[lindex $v_dev_parameters_list 7]
    set v_dev_timing         0x[lindex $v_dev_parameters_list 9]
    set v_dev_non_default      [lindex $v_dev_parameters_list 14]
    set v_dev_xspi_en          [lindex $v_dev_parameters_list 15]
    set v_dev_wrap_en          [lindex $v_dev_parameters_list 16]

    #
    # set what paraeters are visible
    #
    if {$v_dev_name == "none"} {
       set_parameter_property     GUI_DEV${dev}_TYPE         VISIBLE false
       set_parameter_property     GUI_DEV${dev}_SIZE         VISIBLE false
       set_parameter_property     GUI_DEV${dev}_SPEED        VISIBLE false
       set_parameter_property     GUI_DEV${dev}_DQ_WIDTH     VISIBLE false
       set_parameter_property     GUI_DEV${dev}_WRAP_SUPPORT VISIBLE false
       set_parameter_property     GUI_DEV${dev}_NON_DEFAULT  VISIBLE false
       set_parameter_property     GUI_DEV${dev}_T_CSS        VISIBLE false
       set_parameter_property     GUI_DEV${dev}_T_CSH        VISIBLE false
       set_parameter_property     GUI_DEV${dev}_T_RWR        VISIBLE false
       set_parameter_property     GUI_DEV${dev}_T_ACC        VISIBLE false
       set_parameter_property     GUI_DEV${dev}_T_ACC        VISIBLE false
    } else {
       set_parameter_property     GUI_DEV${dev}_TYPE         VISIBLE true
       set_parameter_property     GUI_DEV${dev}_SIZE         VISIBLE true
       set_parameter_property     GUI_DEV${dev}_SPEED        VISIBLE true
       set_parameter_property     GUI_DEV${dev}_DQ_WIDTH     VISIBLE true
       set_parameter_property     GUI_DEV${dev}_WRAP_SUPPORT VISIBLE true
       set_parameter_property     GUI_DEV${dev}_NON_DEFAULT  VISIBLE true
       set_parameter_property     GUI_DEV${dev}_T_CSS        VISIBLE true
       set_parameter_property     GUI_DEV${dev}_T_CSH        VISIBLE true
       set_parameter_property     GUI_DEV${dev}_T_RWR        VISIBLE true
       set_parameter_property     GUI_DEV${dev}_T_ACC        VISIBLE true
       set_parameter_property     GUI_DEV${dev}_T_ACC        VISIBLE true
    }

    #
    # CFG Parameters
    #
       set_parameter_value     GUI_DEV${dev}_NAME         $v_dev_name
       set_parameter_value     GUI_DEV${dev}_TYPE         $v_dev_type
       set_parameter_value     GUI_DEV${dev}_SIZE         [expr $v_dev_size/8]
       set_parameter_value     GUI_DEV${dev}_SPEED        $v_dev_speed
       set_parameter_value     GUI_DEV${dev}_DQ_WIDTH     $v_dev_dq_width
       set_parameter_value     GUI_DEV${dev}_WRAP_SUPPORT $v_dev_wrap_en
       set_parameter_value     GUI_DEV${dev}_NON_DEFAULT  $v_dev_non_default
       set_parameter_value     GUI_DEV${dev}_IS_XSPI      $v_dev_xspi_en


    #
    #check whether device is flash
    #
    if {($v_dev_config & 0x1) == 0} {
       set_parameter_value     GUI_DEV${dev}_IS_FLASH  1
    } else {
       set_parameter_value     GUI_DEV${dev}_IS_FLASH  0
    }


    #
    # Timing Parameters
    #
    set_parameter_value     GUI_DEV${dev}_T_CSS    [expr ($v_dev_timing  >> 0 ) & 0xf]
    set_parameter_value     GUI_DEV${dev}_T_CSH    [expr ($v_dev_timing  >> 4 ) & 0xf]
    set_parameter_value     GUI_DEV${dev}_T_RWR    [expr (($v_dev_timing >> 8 ) + 3) & 0x1F]
    set_parameter_value     GUI_DEV${dev}_T_ACC    [expr ($v_dev_timing  >> 16) & 0x1f]

    #
    #HDL paramters
    #
    set_parameter_value g_dev${dev}_config       32'h[lindex $v_dev_parameters_list 7]
    set_parameter_value g_dev${dev}_config_ext   32'h[lindex $v_dev_parameters_list 8]
    set_parameter_value g_dev${dev}_timing       32'h[lindex $v_dev_parameters_list 9]
    set_parameter_value g_dev${dev}_xspi_rw_cmd  64'h[lindex $v_dev_parameters_list 10]
    set_parameter_value g_dev${dev}_config_ext2  64'h[lindex $v_dev_parameters_list 11]
    set_parameter_value g_dev${dev}_config_ext3  64'h[lindex $v_dev_parameters_list 12]


    #
    # Set SPI-DDR boot config
    #
    ##set_parameter_value g_reg_burst_support      0

    if {$v_dev_xspi_en == 1} {
        set_parameter_property  GUI_DEV${dev}_SPI_BOOT  VISIBLE   true

        if {[get_parameter_value GUI_DEV${dev}_SPI_BOOT] } {
          set_parameter_value g_dev${dev}_config_spi   64'h[lindex $v_dev_parameters_list 13]
          set_parameter_value g_reg_burst_support      1

          #
          # Set Generic Profile
          #
          if {$v_dev_is_generic == 1} {
            set_parameter_value g_reg_generic_mode    2
          }
        } else {
          set_parameter_value g_dev${dev}_config_spi   64'h0000000000000000
        }
    } else {
        set_parameter_property  GUI_DEV${dev}_SPI_BOOT  VISIBLE   false

        set_parameter_value g_dev${dev}_config_spi   64'h0000000000000000
    }
}

#-------------------------------------------------------------------------------
#function : IAVS0 configuration Menu
#-------------------------------------------------------------------------------
proc proc_add_iavs_config {NAME } {
    add_display_item "" "${NAME} Configuration" GROUP tab


    add_display_item "${NAME} Configuration" "${NAME}: Ingress Avalon port stage" GROUP ""
    add_display_item "${NAME} Configuration" "${NAME}: Ingress Avalon address/data" GROUP ""
    add_display_item "${NAME} Configuration" "${NAME}: Burst converter and address decoder stage" GROUP ""
    add_display_item "${NAME} Configuration" "${NAME}: Ingress Avalon return stage" GROUP ""

    add_display_item "${NAME}: Ingress Avalon port stage"                 GUI_${NAME}_WRITE_MODE          PARAMETER ""
    add_display_item "${NAME}: Ingress Avalon port stage"                 GUI_${NAME}_BYTEENABLE          PARAMETER ""
    add_display_item "${NAME}: Ingress Avalon port stage"                 GUI_${NAME}_PORT_MODE           PARAMETER ""
    add_display_item "${NAME}: Ingress Avalon port stage"                 GUI_${NAME}_REG_WDATA           PARAMETER ""

    add_display_item "${NAME}: Ingress Avalon address/data"               GUI_${NAME}_ADDR_WIDTH          PARAMETER ""
    add_display_item "${NAME}: Ingress Avalon address/data"               GUI_${NAME}_ADDR_UNITS          PARAMETER ""
    add_display_item "${NAME}: Ingress Avalon address/data"               GUI_${NAME}_DATA_WIDTH          PARAMETER ""

    add_display_item "${NAME}: Burst converter and address decoder stage" GUI_${NAME}_MAX_BURST_IN_WORDS  PARAMETER ""
    add_display_item "${NAME}: Burst converter and address decoder stage" GUI_${NAME}_LINEWRAP_BURST      PARAMETER ""
    add_display_item "${NAME}: Burst converter and address decoder stage" GUI_${NAME}_BURST_ON_BURST_BOUNDARIES_ONLY PARAMETER ""
    add_display_item "${NAME}: Burst converter and address decoder stage" GUI_${NAME}_BURST_ON_BURST_BOUNDARIES_AUTO PARAMETER ""

    add_display_item "${NAME}: Ingress Avalon return stage"                GUI_${NAME}_REG_RDATA           PARAMETER ""
    add_display_item "${NAME}: Ingress Avalon return stage"                GUI_${NAME}_USE_RESPONSE        PARAMETER ""

    proc_add_parameter     GUI_${NAME}_WRITE_MODE BOOLEAN true "" false  "AV Configuration"
    set_parameter_property GUI_${NAME}_WRITE_MODE         DISPLAY_NAME "Enable Avalon  write  capability" ;
    set_parameter_property GUI_${NAME}_WRITE_MODE         DESCRIPTION  \
        "Enable the Avalon  write  signals on this slave interface."

    proc_add_parameter     GUI_${NAME}_BYTEENABLE BOOLEAN true "" false  "AV Configuration"
    set_parameter_property GUI_${NAME}_BYTEENABLE DISPLAY_NAME "Enable Avalon  byte-enable  capability" ;# This is the GUI label that appears to the left of this parameter.
    set_parameter_property GUI_${NAME}_BYTEENABLE DESCRIPTION \
        "Enable the Avalon  byte-enable  signals on this slave interface. \
        The Nios II instruction bus master does NOT employ byte-enable signals. \
        In contrast, the Nios II data bus master does employs byte-enable signals."

    proc_add_parameter     GUI_${NAME}_PORT_MODE String "Read/Write" "" true  "AV Configuration"
    set_parameter_property GUI_${NAME}_PORT_MODE DISPLAY_NAME "Access capabilities"  ;# This is the GUI label that appears to the left of this parameter.
    set_parameter_property GUI_${NAME}_PORT_MODE DESCRIPTION "Informs the developer if this ingress avalon slave port supports read and/or write capabilities"

    proc_add_parameter     GUI_${NAME}_REG_WDATA BOOLEAN false "" false  "AV Configuration"
    set_parameter_property GUI_${NAME}_REG_WDATA         DISPLAY_NAME "Register Avalon write data path (generally recommended for high clock speed designs)" ;
    set_parameter_property GUI_${NAME}_REG_WDATA         DESCRIPTION  \
        "When enabled, the value of the Avalon wdata signal is cached in a 32-bit wide register.\
        Registration of the write signal can improve place-and-route results and may enable higher clock speed designs.\
        Registration of the write signal does NOT reduce write memory transfer request performance.\
        Turn write registration off to reduce circuit area requirements.\
        High Avalon clock-speed designs typically enable registration of the  wdata  signal, where as lower-clock speed Avalon designs typically do not."

    proc_add_parameter     GUI_${NAME}_REG_RDATA BOOLEAN false "" false  "AV Configuration"
    set_parameter_property GUI_${NAME}_REG_RDATA         DISPLAY_NAME "Register Avalon read data path (sometimes used to increase top clock speeds)" ;
    set_parameter_property GUI_${NAME}_REG_RDATA         DESCRIPTION  \
       "When enabled, the Avalon  rdata  signal is driven by a 32-bit wide register.\
        Registration can improve place-and-route results and may enable higher clock speed designs.\
        Registration delays the read data on the  rdata  signal by 1 clock cycle.\
        Turn read registration off to reduce circuit area requirements and improve software performance.\
        Typically most designs will NOT need to register the  rdata  signal."

    proc_add_parameter     GUI_${NAME}_USE_RESPONSE BOOLEAN false "" false  "AV Configuration"
    set_parameter_property GUI_${NAME}_USE_RESPONSE         DISPLAY_NAME "Use Avalon transaction responses" ;
#    set_parameter_property GUI_${NAME}_USE_RESPONSE         DESCRIPTION  \
#        "When enabled, the Avalon response signal will be added to the slave interface on this port. If you do not know why you need this, do not enable this capability."
    set_parameter_property GUI_${NAME}_USE_RESPONSE         DESCRIPTION  \
        "Please contact S/Labs if you require the ability to enable the Avalon response signal to the slave interface on this port. The vast majority of projects do not require this capability."
    set_parameter_property GUI_${NAME}_USE_RESPONSE         ENABLED  false


    proc_add_parameter     GUI_${NAME}_MAX_BURST_IN_WORDS INTEGER 8 "" false  "AV Configuration"

    set_parameter_property GUI_${NAME}_MAX_BURST_IN_WORDS ALLOWED_RANGES {1,8,16,32,64,128}
    set_parameter_property GUI_${NAME}_MAX_BURST_IN_WORDS DISPLAY_NAME "maxBurstSize (in words)"  ;# This is the GUI label that appears to the left of this parameter.
    set_parameter_property GUI_${NAME}_MAX_BURST_IN_WORDS DESCRIPTION \
        "Select the maximum burst length of read and/or write burst memory transfer requests on this port in words. \
        For best performance and lowest circuit area implementations, the value of maxBurstSize must be set to the value of the largest maxBurstSize of all Avalon bus masters interfaces that are connected to this slave interface."

    proc_add_parameter     GUI_${NAME}_LINEWRAP_BURST    BOOLEAN true "" false "AV Configuration"
    set_parameter_property GUI_${NAME}_LINEWRAP_BURST    ALLOWED_RANGES {"0:false" "1:true"}
    set_parameter_property GUI_${NAME}_LINEWRAP_BURST    DISPLAY_NAME "linewrapBursts" ;# This is the GUI label that appears to the left of this parameter.
    set_parameter_property GUI_${NAME}_LINEWRAP_BURST    DESCRIPTION \
        "If one or more of the Altera Avalon bus master interfaces connected to this slave port employs linewrapBursts=true, then set the \
        linewrapBursts of this slave port to true.  If none of the Altera Avalon bus master interfaces connected to this slave port employ \
        linewrapBursts=true, then we recommend setting the value of linewrapBursts for this slave port to false. \
        In Nios II applications, the Nios II/f L1 instruction cache bus master employs linewrapBursts=true, where as the L1 data cache bus master employs linewrapBursts=false. \
        In contrast, the Nios II/e does not have any internal L1 caches.\
        For Nios II/e applications we strongly recommend S/Labs range of burst capable caches, including our CA-CMS-T003 product."

    proc_add_parameter     GUI_${NAME}_BURST_ON_BURST_BOUNDARIES_ONLY    BOOLEAN false "" false "AV Configuration"
    set_parameter_property GUI_${NAME}_BURST_ON_BURST_BOUNDARIES_ONLY    ALLOWED_RANGES {"0:false" "1:true"}
    set_parameter_property GUI_${NAME}_BURST_ON_BURST_BOUNDARIES_ONLY    DISPLAY_NAME "burstOnBurstBoundariesOnly" ;# This is the GUI label that appears to the left of this parameter.
    set_parameter_property GUI_${NAME}_BURST_ON_BURST_BOUNDARIES_ONLY    DESCRIPTION \
        "Set the value of burstOnBurstBoundariesOnly=true for the slave port when all bus master interfaces connected to this port set the value of their burstOnBurstBoundariesOnly=true. \
        Otherwise, set the value of burstOnBurstBoundariesOnly to false for the slave port.  Setting burstOnBurstBoundariesOnly=true results in lower-circuit area implementations.\
        If you do not know which value to use, set burstOnBurstBoundaries=false."

    proc_add_parameter     GUI_${NAME}_BURST_ON_BURST_BOUNDARIES_AUTO    BOOLEAN true "" true "AV Configuration"
    set_parameter_property GUI_${NAME}_BURST_ON_BURST_BOUNDARIES_AUTO    ALLOWED_RANGES {"0:false" "1:true"}
    set_parameter_property GUI_${NAME}_BURST_ON_BURST_BOUNDARIES_AUTO    DISPLAY_NAME "burstOnBurstBoundariesOnly" ;# This is the GUI label that appears to the left of this parameter.
    set_parameter_property GUI_${NAME}_BURST_ON_BURST_BOUNDARIES_AUTO    DESCRIPTION \
        "Set the value of burstOnBurstBoundariesOnly=true for the slave port when all bus master interfaces connected to this port set the value of their burstOnBurstBoundariesOnly=true. \
        Otherwise, set the value of burstOnBurstBoundariesOnly to false for the slave port.  Setting burstOnBurstBoundariesOnly=true results in lower-circuit area implementations."

    proc_add_parameter     GUI_${NAME}_DATA_WIDTH INTEGER 32 "bits" false  "AV Configuration"
    set_parameter_property GUI_${NAME}_DATA_WIDTH         ALLOWED_RANGES {32}
    set_parameter_property GUI_${NAME}_DATA_WIDTH         DISPLAY_NAME "Word width" ;# This is the GUI label that appears to the left of this parameter.
    set_parameter_property GUI_${NAME}_DATA_WIDTH         DESCRIPTION "The data-path width of this Avalon slave interface"

    proc_add_parameter     GUI_${NAME}_ADDR_WIDTH INTEGER 10 "bits" true "AV Configuration"
    set_parameter_property GUI_${NAME}_ADDR_WIDTH         DISPLAY_NAME "Address width" ;# This is the GUI label that appears to the left of this parameter.
    set_parameter_property GUI_${NAME}_ADDR_WIDTH         DESCRIPTION "The width of the  address signal  of this Avalon slave interface"

    proc_add_parameter     GUI_${NAME}_ADDR_UNITS String "Words" "" false "AV Configuration"
    set_parameter_property GUI_${NAME}_ADDR_UNITS         DISPLAY_NAME "Address units" ;# This is the GUI label that appears to the left of this parameter.
    set_parameter_property GUI_${NAME}_ADDR_UNITS         DESCRIPTION  "Address units"
    set_parameter_property GUI_${NAME}_ADDR_UNITS         ENABLED      false ;

}

#-------------------------------------------------------------------------------
#function : AXI configuration Menu
#-------------------------------------------------------------------------------
proc proc_add_axi_config {NAME } {
    add_display_item "" "${NAME} Configuration" GROUP tab


    add_display_item "${NAME} Configuration" "${NAME}: Ingress AXI port stage" GROUP ""
    add_display_item "${NAME} Configuration" "${NAME}: Ingress AXI address/data" GROUP ""
    add_display_item "${NAME} Configuration" "${NAME}: Burst converter and address decoder stage" GROUP ""
    add_display_item "${NAME} Configuration" "${NAME}: Ingress AXI return stage" GROUP ""

    add_display_item "${NAME}: Ingress AXI port stage"                 GUI_${NAME}_WRITE_MODE          PARAMETER ""
    add_display_item "${NAME}: Ingress AXI port stage"                 GUI_${NAME}_BYTEENABLE          PARAMETER ""
    add_display_item "${NAME}: Ingress AXI port stage"                 GUI_${NAME}_PORT_MODE           PARAMETER ""
    add_display_item "${NAME}: Ingress AXI port stage"                 GUI_${NAME}_REG_WDATA           PARAMETER ""
    add_display_item "${NAME}: Ingress AXI port stage"                 GUI_${NAME}_ID_WIDTH            PARAMETER ""

    add_display_item "${NAME}: Ingress AXI address/data"               GUI_${NAME}_ADDR_WIDTH          PARAMETER ""
    add_display_item "${NAME}: Ingress AXI address/data"               GUI_${NAME}_ADDR_UNITS          PARAMETER ""
    add_display_item "${NAME}: Ingress AXI address/data"               GUI_${NAME}_DATA_WIDTH          PARAMETER ""

    add_display_item "${NAME}: Burst converter and address decoder stage" GUI_${NAME}_MAX_BURST_IN_WORDS  PARAMETER ""
    add_display_item "${NAME}: Burst converter and address decoder stage" GUI_${NAME}_LINEWRAP_BURST      PARAMETER ""
    add_display_item "${NAME}: Burst converter and address decoder stage" GUI_${NAME}_BURST_ON_BURST_BOUNDARIES_ONLY PARAMETER ""
    add_display_item "${NAME}: Burst converter and address decoder stage" GUI_${NAME}_BURST_ON_BURST_BOUNDARIES_AUTO PARAMETER ""

    add_display_item "${NAME}: Ingress AXI return stage"                GUI_${NAME}_REG_RDATA           PARAMETER ""

    proc_add_parameter     GUI_${NAME}_WRITE_MODE BOOLEAN true "" false  "AV Configuration"
    set_parameter_property GUI_${NAME}_WRITE_MODE         DISPLAY_NAME "Enable AXI  write  capability" ;
    set_parameter_property GUI_${NAME}_WRITE_MODE         DESCRIPTION  \
        "Enable the AXI  write  signals on this slave interface."

    proc_add_parameter     GUI_${NAME}_BYTEENABLE BOOLEAN true "" false  "AV Configuration"
    set_parameter_property GUI_${NAME}_BYTEENABLE DISPLAY_NAME "Enable AXI  byte-enable  capability" ;# This is the GUI label that appears to the left of this parameter.
    set_parameter_property GUI_${NAME}_BYTEENABLE DESCRIPTION \
        "Enable the AXI  byte-enable  signals on this slave interface. \
        The Nios II instruction bus master does NOT employ byte-enable signals. \
        In contrast, the Nios II data bus master does employs byte-enable signals."

    proc_add_parameter     GUI_${NAME}_PORT_MODE String "Read/Write" "" true  "AV Configuration"
    set_parameter_property GUI_${NAME}_PORT_MODE DISPLAY_NAME "Access capabilities"  ;# This is the GUI label that appears to the left of this parameter.
    set_parameter_property GUI_${NAME}_PORT_MODE DESCRIPTION "Informs the developer if this ingress AXI slave port supports read and/or write capabilities"

    proc_add_parameter     GUI_${NAME}_REG_WDATA BOOLEAN false "" false  "AV Configuration"
    set_parameter_property GUI_${NAME}_REG_WDATA         DISPLAY_NAME "Register AXI write data path (generally recommended for high clock speed designs)" ;
    set_parameter_property GUI_${NAME}_REG_WDATA         DESCRIPTION  \
        "When enabled, the value of the AXI wdata signal is cached in a 32-bit wide register.\
        Registration of the write signal can improve place-and-route results and may enable higher clock speed designs.\
        Registration of the write signal does NOT reduce write memory transfer request performance.\
        Turn write registration off to reduce circuit area requirements.\
        High AXI clock-speed designs typically enable registration of the  wdata  signal, where as lower-clock speed AXI designs typically do not."

    proc_add_parameter     GUI_${NAME}_ID_WIDTH INTEGER 4 "" false  "AV Configuration"
    set_parameter_property GUI_${NAME}_ID_WIDTH ALLOWED_RANGES {1:31}
    set_parameter_property GUI_${NAME}_ID_WIDTH DISPLAY_NAME "AXI ID width"  ;# This is the GUI label that appears to the left of this parameter.
    set_parameter_property GUI_${NAME}_ID_WIDTH DESCRIPTION \
        "Select the AXI ID width"


    proc_add_parameter     GUI_${NAME}_REG_RDATA BOOLEAN false "" false  "AV Configuration"
    set_parameter_property GUI_${NAME}_REG_RDATA         DISPLAY_NAME "Register AXI read data path (sometimes used to increase top clock speeds)" ;
    set_parameter_property GUI_${NAME}_REG_RDATA         DESCRIPTION  \
       "When enabled, the AXI  rdata  signal is driven by a 32-bit wide register.\
        Registration can improve place-and-route results and may enable higher clock speed designs.\
        Registration delays the read data on the  rdata  signal by 1 clock cycle.\
        Turn read registration off to reduce circuit area requirements and improve software performance.\
        Typically most designs will NOT need to register the  rdata  signal."


    proc_add_parameter     GUI_${NAME}_MAX_BURST_IN_WORDS INTEGER 256 "" false  "AV Configuration"

    set_parameter_property GUI_${NAME}_MAX_BURST_IN_WORDS ALLOWED_RANGES {1,8,16,32,64,128,256}
    set_parameter_property GUI_${NAME}_MAX_BURST_IN_WORDS DISPLAY_NAME "maxBurstSize (in words)"  ;# This is the GUI label that appears to the left of this parameter.
    set_parameter_property GUI_${NAME}_MAX_BURST_IN_WORDS DESCRIPTION \
        "Select the maximum burst length of read and/or write burst memory transfer requests on this port in words. \
        For best performance and lowest circuit area implementations, the value of maxBurstSize must be set to the value of the largest maxBurstSize of all AXI bus masters interfaces that are connected to this slave interface."

    proc_add_parameter     GUI_${NAME}_LINEWRAP_BURST    BOOLEAN true "" false "AV Configuration"
    set_parameter_property GUI_${NAME}_LINEWRAP_BURST    ALLOWED_RANGES {"0:false"}
    set_parameter_property GUI_${NAME}_LINEWRAP_BURST    DISPLAY_NAME "linewrapBursts" ;# This is the GUI label that appears to the left of this parameter.
    set_parameter_property GUI_${NAME}_LINEWRAP_BURST    VISIBLE        FALSE
    set_parameter_property GUI_${NAME}_LINEWRAP_BURST    DESCRIPTION \
        "If one or more of the Altera AXI bus master interfaces connected to this slave port employs linewrapBursts=true, then set the \
        linewrapBursts of this slave port to true.  If none of the Altera AXI bus master interfaces connected to this slave port employ \
        linewrapBursts=true, then we recommend setting the value of linewrapBursts for this slave port to false. \
        In Nios II applications, the Nios II/f L1 instruction cache bus master employs linewrapBursts=true, where as the L1 data cache bus master employs linewrapBursts=false. \
        In contrast, the Nios II/e does not have any internal L1 caches.\
        For Nios II/e applications we strongly recommend S/Labs range of burst capable caches, including our CA-CMS-T003 product."

    proc_add_parameter     GUI_${NAME}_BURST_ON_BURST_BOUNDARIES_ONLY    BOOLEAN false "" false "AV Configuration"
    set_parameter_property GUI_${NAME}_BURST_ON_BURST_BOUNDARIES_ONLY    ALLOWED_RANGES {"0:false" "1:true"}
    set_parameter_property GUI_${NAME}_BURST_ON_BURST_BOUNDARIES_ONLY    DISPLAY_NAME "burstOnBurstBoundariesOnly" ;# This is the GUI label that appears to the left of this parameter.
    set_parameter_property GUI_${NAME}_BURST_ON_BURST_BOUNDARIES_ONLY    VISIBLE        FALSE
    set_parameter_property GUI_${NAME}_BURST_ON_BURST_BOUNDARIES_ONLY    DESCRIPTION \
        "Set the value of burstOnBurstBoundariesOnly=true for the slave port when all bus master interfaces connected to this port set the value of their burstOnBurstBoundariesOnly=true. \
        Otherwise, set the value of burstOnBurstBoundariesOnly to false for the slave port.  Setting burstOnBurstBoundariesOnly=true results in lower-circuit area implementations.\
        If you do not know which value to use, set burstOnBurstBoundaries=false."

    proc_add_parameter     GUI_${NAME}_BURST_ON_BURST_BOUNDARIES_AUTO    BOOLEAN true "" true "AV Configuration"
    set_parameter_property GUI_${NAME}_BURST_ON_BURST_BOUNDARIES_AUTO    ALLOWED_RANGES {"0:false" "1:true"}
    set_parameter_property GUI_${NAME}_BURST_ON_BURST_BOUNDARIES_AUTO    VISIBLE        FALSE
    set_parameter_property GUI_${NAME}_BURST_ON_BURST_BOUNDARIES_AUTO    DISPLAY_NAME "burstOnBurstBoundariesOnly" ;# This is the GUI label that appears to the left of this parameter.
    set_parameter_property GUI_${NAME}_BURST_ON_BURST_BOUNDARIES_AUTO    DESCRIPTION \
        "Set the value of burstOnBurstBoundariesOnly=true for the slave port when all bus master interfaces connected to this port set the value of their burstOnBurstBoundariesOnly=true. \
        Otherwise, set the value of burstOnBurstBoundariesOnly to false for the slave port.  Setting burstOnBurstBoundariesOnly=true results in lower-circuit area implementations."

    proc_add_parameter     GUI_${NAME}_DATA_WIDTH INTEGER 32 "bits" true  "AV Configuration"
    set_parameter_property GUI_${NAME}_DATA_WIDTH         ALLOWED_RANGES {32 64 128}
    set_parameter_property GUI_${NAME}_DATA_WIDTH         DISPLAY_NAME "Word width" ;# This is the GUI label that appears to the left of this parameter.
    set_parameter_property GUI_${NAME}_DATA_WIDTH         DESCRIPTION "The data-path width of this AXI slave interface"

    proc_add_parameter     GUI_${NAME}_ADDR_WIDTH INTEGER 10 "bits" true "AV Configuration"
    set_parameter_property GUI_${NAME}_ADDR_WIDTH         DISPLAY_NAME "Address width" ;# This is the GUI label that appears to the left of this parameter.
    set_parameter_property GUI_${NAME}_ADDR_WIDTH         DESCRIPTION "The width of the  address signal  of this AXI slave interface"

    proc_add_parameter     GUI_${NAME}_ADDR_UNITS String "Words" "" false "AV Configuration"
    set_parameter_property GUI_${NAME}_ADDR_UNITS         DISPLAY_NAME "Address units" ;# This is the GUI label that appears to the left of this parameter.
    set_parameter_property GUI_${NAME}_ADDR_UNITS         DESCRIPTION  "Address units"
    set_parameter_property GUI_${NAME}_ADDR_UNITS         ENABLED      false ;

}

#-------------------------------------------------------------------------------
# connection point Conduit
#-------------------------------------------------------------------------------
proc proc_Add_Memory_Conduit_IO {v_num_chipselect  phylite_interface} {


    set_interface_property     i_hbus_clk_0        ENABLED true


    if {$phylite_interface} {
      set_interface_property     i_hbus_clk_90       ENABLED false

    } else {
      set_interface_property     i_hbus_clk_90       ENABLED true

    }

    #
    #Memory Signal Interface
    set_interface_property Conduit_IO associatedClock     clock
    set_interface_property Conduit_IO associatedReset     ""
    set_interface_property Conduit_IO ENABLED             true
    set_interface_property Conduit_IO EXPORT_OF           ""
    set_interface_property Conduit_IO PORT_NAME_MAP       ""
    set_interface_property Conduit_IO CMSIS_SVD_VARIABLES ""
    set_interface_property Conduit_IO SVD_ADDRESS_GROUP   ""

    #Memory signals
    add_interface_port Conduit_IO MB_CLK0    MB0_CLK0     Output 1
    add_interface_port Conduit_IO MB_CLK0n   MB0_CLK0n    Output 1
    add_interface_port Conduit_IO MB_CLK1    MB0_CLK1     Output 1
    add_interface_port Conduit_IO MB_CLK1n   MB0_CLK1n    Output 1
    add_interface_port Conduit_IO MB_CS0n    MB0_CS0n     Output 1
    add_interface_port Conduit_IO MB_CS1n    MB0_CS1n     Output 1
    add_interface_port Conduit_IO MB_RWDS    MB0_RWDS     Bidir  g_dqs_width
    add_interface_port Conduit_IO MB_DQ      MB0_DQ       Bidir  g_dqin_width

    add_interface_port Conduit_IO MB_RSTn    MB0_RSTn     Output 1
    add_interface_port Conduit_IO MB_WPn     MB0_WPn      Output 1
    add_interface_port Conduit_IO MB_RSTOn   MB0_RSTOn    Input  1
    add_interface_port Conduit_IO MB_INTn    MB0_INTn     Input  1
    add_interface_port Conduit_IO MB_ECC     MB0_ECC      Input  1

    #PHYLite interface
    add_interface_port Conduit_IO wdata_oe          wdata_oe           Output  g_dqin_width
    add_interface_port Conduit_IO wdata_from_core   wdata_from_core    Output  (2*g_dqin_width)

    add_interface_port Conduit_IO wds_oe            wds_oe             Output  g_dqs_width
    add_interface_port Conduit_IO wds_from_core     wds_from_core      Output  (2*g_dqs_width)

    add_interface_port Conduit_IO clk_out_mux       clk_out_mux        Output  2
    add_interface_port Conduit_IO clkn_out_mux      clkn_out_mux       Output  2

    add_interface_port Conduit_IO cs0n_out          cs0n_out           Output  1
    add_interface_port Conduit_IO cs1n_out          cs1n_out           Output  1

    add_interface_port Conduit_IO rdata_en          rdata_en           Output  1
    add_interface_port Conduit_IO rdata_valid       rdata_valid        Input   1
    add_interface_port Conduit_IO rdata_to_core     rdata_to_core      Input   (2*g_dqin_width)

    if {$phylite_interface} {
      set_port_property MB_CLK0          TERMINATION true
      set_port_property MB_CLK0n         TERMINATION true
      set_port_property MB_CLK1          TERMINATION true
      set_port_property MB_CLK1n         TERMINATION true
      set_port_property MB_CS0n          TERMINATION true
      set_port_property MB_CS1n          TERMINATION true
      set_port_property MB_RWDS          TERMINATION true
      set_port_property MB_DQ            TERMINATION true

      set_port_property wdata_oe         TERMINATION false
      set_port_property wdata_from_core  TERMINATION false

      set_port_property wds_oe           TERMINATION false
      set_port_property wds_from_core    TERMINATION false

      set_port_property clk_out_mux      TERMINATION false
      set_port_property clkn_out_mux     TERMINATION false

      set_port_property cs0n_out         TERMINATION false
      set_port_property cs1n_out         TERMINATION false


      set_port_property rdata_en         TERMINATION false
      set_port_property rdata_valid      TERMINATION false
      set_port_property rdata_to_core    TERMINATION false
    } else {
      set_port_property MB_CLK0          TERMINATION false
      set_port_property MB_CLK0n         TERMINATION false
      set_port_property MB_CLK1          TERMINATION false
      set_port_property MB_CLK1n         TERMINATION false
      set_port_property MB_CS0n          TERMINATION false
      set_port_property MB_CS1n          TERMINATION false
      set_port_property MB_RWDS          TERMINATION false
      set_port_property MB_DQ            TERMINATION false

      set_port_property wdata_oe         TERMINATION true
      set_port_property wdata_from_core  TERMINATION true

      set_port_property wds_oe           TERMINATION true
      set_port_property wds_from_core    TERMINATION true

      set_port_property clk_out_mux      TERMINATION true
      set_port_property clkn_out_mux     TERMINATION true

      set_port_property cs0n_out         TERMINATION true
      set_port_property cs1n_out         TERMINATION true

      set_port_property rdata_en         TERMINATION true
      set_port_property rdata_valid      TERMINATION true
      set_port_property rdata_to_core    TERMINATION true
    }
}

proc proc_add_axi4_slave {name } {
    add_interface          $name    axi4 slave
    set_interface_property $name readAcceptanceCapability     1
    set_interface_property $name writeAcceptanceCapability    1
    set_interface_property $name combinedAcceptanceCapability 1
    set_interface_property $name readDataReorderingDepth      1
    set_interface_property $name ENABLED                      true

}

proc proc_add_axi_conduit { name } {

    add_interface_port $name ${name}_awid     awid     Input   g_iavs0_id_width
    add_interface_port $name ${name}_awaddr   awaddr   Input   g_iavs0_addr_width
    add_interface_port $name ${name}_awlen    awlen    Input   8
    add_interface_port $name ${name}_awsize   awsize   Input   3
    add_interface_port $name ${name}_awburst  awburst  Input   2
    add_interface_port $name ${name}_awlock   awlock   Input   1
#   add_interface_port $name ${name}_awcache  awcache  Input   4
    add_interface_port $name ${name}_awprot   awprot   Input   3
#   add_interface_port $name ${name}_awuser   awuser   Input   5
#   add_interface_port $name ${name}_awqos    awqos    Input   4
#   add_interface_port $name ${name}_awregion awregion Input   4
    add_interface_port $name ${name}_awvalid  awvalid  Input   1
    add_interface_port $name ${name}_awready  awready  Output  1
    add_interface_port $name ${name}_wdata    wdata    Input   g_iavs0_data_width
    add_interface_port $name ${name}_wstrb    wstrb    Input   g_iavs0_av_numsymbols
    add_interface_port $name ${name}_wlast    wlast    Input   1
    add_interface_port $name ${name}_wvalid   wvalid   Input   1
#   add_interface_port $name ${name}_wuser    wuser    Input   8
    add_interface_port $name ${name}_wready   wready   Output  1
    add_interface_port $name ${name}_bid      bid      Output  g_iavs0_id_width
    add_interface_port $name ${name}_bresp    bresp    Output  2
#  add_interface_port $name ${name}_buser    buser    Output  8
    add_interface_port $name ${name}_bvalid   bvalid   Output  1
    add_interface_port $name ${name}_bready   bready   Input   1
    add_interface_port $name ${name}_arid     arid     Input   g_iavs0_id_width
    add_interface_port $name ${name}_araddr   araddr   Input   g_iavs0_addr_width
    add_interface_port $name ${name}_arlen    arlen    Input   8
    add_interface_port $name ${name}_arsize   arsize   Input   3
    add_interface_port $name ${name}_arburst  arburst  Input   2
    add_interface_port $name ${name}_arlock   arlock   Input   1
#   add_interface_port $name ${name}_arcache  arcache  Input   4
    add_interface_port $name ${name}_arprot   arprot   Input   3
#   add_interface_port $name ${name}_aruser   aruser   Input   5
#   add_interface_port $name ${name}_arqos    arqos    Input   4
#   add_interface_port $name ${name}_arregion arregion Input   4
    add_interface_port $name ${name}_arvalid  arvalid  Input   1
    add_interface_port $name ${name}_arready  arready  Output  1
    add_interface_port $name ${name}_rid      rid      Output  g_iavs0_id_width
    add_interface_port $name ${name}_rdata    rdata    Output  g_iavs0_data_width
    add_interface_port $name ${name}_rresp    rresp    Output  2
    add_interface_port $name ${name}_rlast    rlast    Output  1
    add_interface_port $name ${name}_rvalid   rvalid   Output  1
    add_interface_port $name ${name}_rready   rready   Input   1
#   add_interface_port $name ${name}_ruser    ruser    Output  8
}


#######################################################################################################################
#populate fragment_list procedure

proc populate_fragment_list {signal_name signal_width mst} {
  set signal_list ""
  for { set z 0 } { $z<$signal_width} { incr z } {
    set z2 [expr $signal_width * ($mst+1) - $z - 1 ]
    append signal_list  "$signal_name@$z2 "
  }

  return $signal_list
}

proc populate_fragment_list2 {signal_name signal_width signal_index} {
  set signal_list ""

  for { set z 0 } { $z<$signal_width} { incr z } {
    set z2 [expr ($signal_width + $signal_index) - $z - 1 ]
    append signal_list  "$signal_name@$z2 "
  }

  return $signal_list
}
